package graphics;

/**
 * 
 * An abstract class that is the superclass for most graphics.  Defines most capabilities that are shared among all Graphic shapes
 *
 * @author <a href="mailto:kozelsky@cse.buffalo.edu">Michael Kozelsky</a>
 *
 * Created on: Jul 28, 2006
 *
 *
 * AbstractGraphic.java
 */
public abstract class AbstractGraphic implements IGraphic {

	/** The Dimension of this graphic * */
	private java.awt.Dimension _dimension;

	/** The location of this graphic * */
	private java.awt.Point _location;

	/**
	 * The container of this graphic, will be defined when it is added to a
	 * container 
	 */
	private IContainer _container;

	/** The degree of rotation of this graphic * */
	private Integer _rotation;

	/** The MouseListeners that this graphic has * */
	private java.util.ArrayList<java.awt.event.MouseListener> _mouseListeners;

	/** The MouseMotionListeners that this graphic has * */
	private java.util.ArrayList<java.awt.event.MouseMotionListener> _mouseMotionListeners;

	/** The Movement of this graphic **/
	private graphics.movement.IGraphicMovement _movement;

	/**
	 * Creates a new instance of AbstractGraphic with a default size (0,0);
	 * location (0,0); and rotation (0)
	 */
	public AbstractGraphic() {
		_dimension = new java.awt.Dimension();
		_location = new java.awt.Point();
		_rotation = 0;
		_mouseListeners = new java.util.ArrayList<java.awt.event.MouseListener>();
		_mouseMotionListeners = new java.util.ArrayList<java.awt.event.MouseMotionListener>();
		_movement = new graphics.movement.RegularMovement();
	}

	/**
	 * Gets the Container of this graphic
	 * 
	 * @return the IContainer that this graphic is contained in. Returns null if
	 *         this does not have an IContainer
	 * @see #setContainer(IContainer container)
	 */
	public IContainer getContainer() {
		return _container;
	}

	/**
	 * Sets the Container of this Graphic. Should be automatically called inside
	 * the addGraphic method of the IContainer that contains this graphic
	 * 
	 * @param container
	 */
	public void setContainer(IContainer container) {
		// System.out.println("setting the container of "+this);
		// if we already have a container, remove this from the container
		if (_container != null) {
			_container.remove(this);
			_container.repaint(this.getBounds());
		}

		// set _container to be what was passed in
		_container = container;
		_container.repaint(this.getBounds());
	}

	/**
	 * This gets the concrete subclasses ready to do the actual painting - it
	 * figures out a rotation, then delegates the drawing to the specific
	 * subclass
	 * 
	 * @see graphics.IGraphic#paint(java.awt.Graphics2D)
	 * @param gs
	 *            The Graphics2D object that draws the shapes on the IContainer
	 */
	public void paint(java.awt.Graphics2D gs) {		
		this.rotateForward(gs);

		// actualPaint should be defined in the subclasses, it will paint the appropriate shape
		this.actualPaint(gs, this.getLocation(), this.getDimension());

		this.rotateBackward(gs);
	}

	/**
	 * Changes the Dimension of this Graphic
	 * 
	 * @see graphics.ISizeable#setDimension(java.awt.Dimension)
	 * @see #getDimension
	 * @param d
	 *            the new Dimension
	 */
	public void setDimension(java.awt.Dimension d) {
		java.awt.Rectangle oldBounds = this.getBounds();
		_dimension = d;
		if (_container != null)
			_container.repaint(this.getBounds().union(oldBounds));
	}

	/**
	 * Gets the dimension of this graphic
	 * 
	 * @see graphics.ISizeable#getDimension()
	 * @see #setDimension
	 * @return the Dimension of this graphic
	 */
	public java.awt.Dimension getDimension() {
		return _dimension;
	}

	/**
	 * Changes the location of this graphic
	 * 
	 * @see graphics.ILocatable#setLocation(java.awt.Point)
	 * @param point
	 *            The new location to put the graphic
	 */
	public void setLocation(java.awt.Point point) {
		java.awt.Rectangle oldBounds = this.getBounds();
		_location = new java.awt.Point(point);
		if (_container != null)
			_container.repaint(this.getBounds().union(oldBounds));
	}

	/**
	 * Gets the location of this graphic
	 * 
	 * @see graphics.ILocatable#getLocation()
	 * @return The current location
	 */
	public java.awt.Point getLocation() {
		return new java.awt.Point(_location);
	}

	/**
	 * Changes the center location of this graphic
	 * 
	 * @see graphics.CenterLocatable#setCenterLocation(java.awt.Point)
	 * @param point
	 *            the location to place the center of this graphic at
	 */
	public void setCenterLocation(java.awt.Point point) {
		this.setLocation(new java.awt.Point(point.x - this.getDimension().width / 2,
				point.y - this.getDimension().height / 2));
	}

	/**
	 * Gets the Center location of this graphic
	 * 
	 * @see graphics.CenterLocatable#getCenterLocation()
	 * @return a Point representing the location at the center of this graphic
	 */
	public java.awt.Point getCenterLocation() {
		return new java.awt.Point(
				this.getLocation().x + (this.getDimension().width / 2),
				this.getLocation().y + (this.getDimension().height / 2));
	}

	/**
	 * Moves the Graphic by a certain vector, according to it's current MovementBehavior
	 * 
	 * @see graphics.IGraphic#move(utilities.Vector)
	 * @param v
	 *            The vector containing the amount to move
	 */
	public utilities.Vector move(utilities.Vector v) {
		return _movement.move(this, v);
	}

	/**
	 * Changes how the graphic moves
	 * @see graphics.IGraphic#setMovement(graphics.movement.IGraphicMovement)
	 * @param movement
	 */
	public void setMovement(graphics.movement.IGraphicMovement movement){
		_movement = movement;
	}
	
	/**
	 * Gets the current movement of the graphic
	 * 	@see graphics.IGraphic#getMovement()
	 * @return this graphics' IGraphicMovement
	 */
	public graphics.movement.IGraphicMovement getMovement(){
		return _movement;
	}
	
	/**
	 * Changes the rotation of this graphic. Should rotate about its center
	 * 
	 * @see graphics.IRotatable#setRotation(java.lang.Integer)
	 * @param degree
	 *            The number of degrees to set the rotation at
	 */
	public void setRotation(Integer degree) {
		java.awt.Rectangle oldBounds = this.getBounds();
		_rotation = degree%360;
		if (_container != null) {
			_container.repaint(this.getBounds().union(oldBounds));
		}
	}

	/**
	 * Gets the current rotation of this graphic
	 * 
	 * @see graphics.IRotatable#getRotation()
	 * @return an Integer (0 through 359) representing the rotation of this
	 *         graphic
	 */
	public Integer getRotation() {
		return _rotation;
	}

	/**
	 * Rotates the graphic by a specific number of degrees
	 * 
	 * @see graphics.IRotatable#rotate(java.lang.Integer)
	 * @param degreesToRotate
	 *            The number of degrees to rotate this graphic
	 */
	public void rotate(Integer degreesToRotate) {
		this.setRotation(this.getRotation() + degreesToRotate);
	}

	/**
	 * This method is used to rotate the Graphics2D back to the start angle of
	 * rotation after drawing the shape (The drawing of the shape is done by
	 * calling actualPaint).
	 * 
	 * @param gs
	 *            The Graphics2D object to rotate
	 */
	private void rotateBackward(java.awt.Graphics2D gs) {
		gs.rotate((this.getRotation() * (Math.PI / 180)) * -1, this
				.getCenterLocation().x, this.getCenterLocation().y);
	}

	/**
	 * This method is used to rotate the Graphics2D around to the appropriate
	 * angle of rotation before drawing the shape (The drawing of the shape is
	 * done by calling actualPaint).
	 * 
	 * @param gs
	 *            The Graphics2D object to rotate
	 */
	private void rotateForward(java.awt.Graphics2D gs) {
		gs.rotate((this.getRotation() * (Math.PI / 180)), this
				.getCenterLocation().x, this.getCenterLocation().y);
	}

	/**
	 * Adds a <code>java.awt.event.MouseListener</code> to this graphic
	 * 
	 * @see graphics.IGraphic#addMouseListener(java.awt.event.MouseListener)
	 * @param ml
	 */
	public void addMouseListener(java.awt.event.MouseListener ml) {
		_mouseListeners.add(ml);
	}

	/**
	 * Gets all the <code>java.awt.event.MouseListeners</code> of this graphic
	 * 
	 * @see graphics.IGraphic#getMouseListeners()
	 * @return a Collection of MouseListeners
	 */
	public java.util.Collection<java.awt.event.MouseListener> getMouseListeners() {
		return _mouseListeners;
	}

	/**
	 * Adds a <code>java.awt.event.MouseMotionListener</code> to this graphic
	 * 
	 * @see graphics.IGraphic#addMouseMotionListener(java.awt.event.MouseMotionListener)
	 * @param mml The <code>java.awt.event.MouseMotionListener</code> to add to the 
	 */
	public void addMouseMotionListener(java.awt.event.MouseMotionListener mml) {
		_mouseMotionListeners.add(mml);
	}

	/**
	 * Gets all the <code>java.awt.event.MouseMotionListeners</code> of this graphic
	 * @see graphics.IGraphic#getMouseMotionListeners()
	 * @return A collection of all the MouseMotionListeners of this graphic
	 */
	public java.util.Collection<java.awt.event.MouseMotionListener> getMouseMotionListeners() {
		return _mouseMotionListeners;
	}

	/**
	 * Returns the BoundingBox of this graphic
	 * @see graphics.IGraphic#getBounds()
	 * @return A java.awt.Rectangle representing the bounding box of this graphic
	 */
	public java.awt.Rectangle getBounds() {		
		return this.getShape().getBounds();
	}
	
	/**
	 * Tests to see if the given point is contained within this shape
	 * 
	 * @see graphics.IGraphic#contains(java.awt.Point)
	 * @param p
	 *            The point to test for containment
	 * @return true if the point is contained in the ellipse, false otherwise
	 */
	public boolean contains(java.awt.Point p) {	
		return this.getShape().contains(p);
	}
	
	/**
	 * Tests to see if two Graphics intersect with each other.
	 * @see graphics.IGraphic#intersects(graphics.IGraphic)
	 * @param g the graphic to test for intersection
	 * @return true if this graphic intersects with g
	 */
	public boolean intersects(IGraphic g){
		//get my area
		java.awt.Shape thisShape = this.getShape();
		java.awt.geom.Area thisArea = new java.awt.geom.Area(thisShape);
		
		//get other guy's area
		java.awt.Shape otherShape = g.getShape();
		java.awt.geom.Area otherArea = new java.awt.geom.Area(otherShape);
		
		//take the intersection of the area
		thisArea.intersect(otherArea);		
		
		//if thisArea is empty, then we did not intersect --> so return the negation
		return !thisArea.isEmpty();
	}
	
	/**
	 * This will apply the rotation to the shape
	 * @param originalShape
	 * @return
	 */
	protected java.awt.Shape applyRotation(java.awt.Shape originalShape){
		//now rotate it
		java.awt.geom.AffineTransform trans = java.awt.geom.AffineTransform.getRotateInstance(
		    	//the rotation in radians, the X coordinate of the center, the Y coordinate of the center
		    	Math.toRadians(this.getRotation()), this.getCenterLocation().x, this.getCenterLocation().y);
			
		java.awt.Shape rotateShape = trans.createTransformedShape(originalShape);
		
		return rotateShape;
	}
	
}
