package graphics;

import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Rectangle;
import java.awt.geom.AffineTransform;

import javax.swing.JComponent;

/**
 * 
 * A Graphical object that acts as a container for other Graphical objects. Add
 * many Graphics to this and then moving, rotating, etc. this graphic will
 * perform the desired action on all the Graphics this contains
 * 
 * @author <a href="mailto:kozelsky@cse.buffalo.edu">Michael Kozelsky</a>
 * 
 * Created on: Jul 28, 2006 TODO: dimensions should be set first, adjusting
 * shapes may not be as expected
 * 
 * CompoundGraphic.java
 */
public class CompoundGraphic extends AbstractGraphic implements IContainer {

	/** The List of Graphics that this compound contains* */
	private java.util.ArrayList<IGraphic> _graphicsList;

	/**
	 * Creates a new instance of an empty CompoundGraphic with a dimension of
	 * (1,1) at point (0,0)
	 */
	public CompoundGraphic() {
		_graphicsList = new java.util.ArrayList<IGraphic>();
		this.setDimension(new java.awt.Dimension(1, 1));
	}

	/**
	 * Adds a graphic to the compound
	 * 
	 * @see graphics.IContainer#add(graphics.IGraphic)
	 * @param g
	 *            The Graphic to add
	 */
	public void add(IGraphic g) {
		_graphicsList.add(g);
		g.setContainer(this);
		if (this.getContainer() != null) {
			this.getContainer().repaint(this.getBounds());
		//	System.out.println("added " + g + ", now repainting");
		}
	}

	/**
	 * Removes a Graphic from the Compound
	 * 
	 * @see graphics.IContainer#remove(graphics.IGraphic)
	 * @param g
	 *            The Graphic to remove
	 */
	public void remove(IGraphic g) {
		_graphicsList.remove(g);
		if (this.getContainer() != null)
			this.getContainer().repaint(this.getBounds());
	}

	/**
	 * Uses the Graphics2D object to Paint the CompoundGraphic at a specified
	 * location and with a specified dimension
	 * 
	 * @see graphics.IGraphic#actualPaint(java.awt.Graphics2D,
	 *      java.awt.Point, java.awt.Dimension)
	 * @param gs
	 *            The Graphics2D object that will do the painting
	 * @param location
	 *            The Location at which to paint the Compound
	 * @param dimension
	 *            The Dimension of the Compound
	 */
	public void actualPaint(java.awt.Graphics2D gs, java.awt.Point location,
			java.awt.Dimension dimension) {
		Graphics offsetGraphics = gs.create(location.x,location.y, dimension.width, dimension.height);
		
		for (IGraphic g : _graphicsList) {
			// paint each graphic with an offset	
			g.paint((Graphics2D)offsetGraphics);
		}
	}

	/**
	 * Forces a repaint of all the Graphics in the same container as the
	 * Compound
	 * 
	 * @see graphics.IContainer#repaint()
	 */
	public void repaint() {
		if (this.getContainer() != null) 
			this.getContainer().repaint();
	}
	
	/**
	 * @see graphics.IContainer#repaint(java.awt.Rectangle)
	 * @param bounds the bounds of which to repaint
	 */
	public void repaint(Rectangle bounds) {
		if(this.getContainer()!=null){
			//translate the bounds we need to repaint by our location,
			//so we can tell our parent to repaint the correct bounds
			bounds.translate(this.getLocation().x, this.getLocation().y);
			this.getContainer().repaint(bounds);
		}
	}

	/**
	 * @param img
	 * @param infoflags
	 * @param x
	 * @param y
	 * @param width
	 * @param height
	 * @return I don't know
	 * @see JComponent#imageUpdate(java.awt.Image, int, int, int, int, int)
	 */
	public boolean imageUpdate(java.awt.Image img, int infoflags, int x, int y,
			int width, int height) {
		// forward request on to this compound's container
		return this.getContainer().imageUpdate(img, infoflags, x, y, width,
				height);
	}

	/**
	 * Sets the dimension of this compound, thus causing all the Graphics in the
	 * compound to be altered. Use with caution. It is much easier if the
	 * Dimensions of CompoundGraphics are set before any Graphics are inserted
	 * into it.
	 * 
	 * @see graphics.ISizeable#setDimension(java.awt.Dimension)
	 * @param newDimension
	 *            The Dimension to set this CompoundGraphic to be
	 */
	@Override
	public void setDimension(java.awt.Dimension newDimension) {
		java.awt.Dimension oldDimension = this.getDimension();
		for (IGraphic g : _graphicsList) {
			this.adjustShape(this, g, oldDimension, newDimension);
		}

		super.setDimension(newDimension);
	}

	/**
	 * Adjusts the specified graphic's size and location based on the
	 * newDimension and it's position relative to the staticShape
	 * 
	 * @param staticShape
	 *            The shape to base the calculations on
	 * @param shapeToAdjust
	 *            The Shape that will be moved/resized
	 * @param oldDimension
	 *            The old dimension of the Compound
	 * @param newDimension
	 *            The dimension that the Compound is to become
	 */
	private void adjustShape(IGraphic staticShape, IGraphic shapeToAdjust,
			java.awt.Dimension oldDimension, java.awt.Dimension newDimension) {
		// System.out.println("staticShape: " + staticShape.getLocation() + ", "
		// + staticShape.getDimension());
		// System.out.println("shapeToMove: " + shapeToMove.getLocation() + ", "
		// + shapeToMove.getDimension());
		// System.out.println("oldDimension: " + oldDimension);
		// System.out.println("newDimension: " + newDimension);

		// move the shape so that it's location stays with the same relative
		// location as before
		double ratioX = (shapeToAdjust.getLocation().x)
				/ (double) oldDimension.width;
		double ratioY = shapeToAdjust.getLocation().y
				/ (double) oldDimension.height;
		// System.out.println("ratioX: " + ratioX + ", ratioY: " + ratioY);

		java.awt.Point newPoint = new java.awt.Point(
				(int) (newDimension.width * ratioX),
				(int) (newDimension.height * ratioY));
		// System.out.println("setting the location of " + shapeToMove + " to be
		// " + newPoint);
		shapeToAdjust.setLocation(newPoint);

		double xOffset = (double) shapeToAdjust.getDimension().width
				/ (double) staticShape.getDimension().width;
		double yOffset = (double) shapeToAdjust.getDimension().height
				/ (double) staticShape.getDimension().height;

		// System.out.println("xoffset: " + xOffset + ", yoffset: " + yOffset);
		shapeToAdjust.setDimension(new java.awt.Dimension(
				(int) (newDimension.width * xOffset),
				(int) (newDimension.height * yOffset)));

		// System.out.println();
	}

	/**
	 * Returns a java.awt.Shape that represents the total sum of the areas of all the Graphics in this Compound's Bounding box.
	 * 
	 * @see graphics.IGraphic#getShape()
	 * @return an empty java.awt.Rectangle
	 */
	public java.awt.Shape getShape() {
		java.awt.geom.Area area =  new java.awt.geom.Area();
		for (IGraphic g: _graphicsList){
			area.add(new java.awt.geom.Area(g.getShape()));
		}
		//now the area contains the sum of all the areas of the graphics it contains
		area.transform(AffineTransform.getTranslateInstance(this.getLocation().x, this.getLocation().y));
		
		//remove all the things that are not in the compound's Clipping area.
		java.awt.Rectangle clip = new java.awt.Rectangle(this.getLocation().x, this.getLocation().y, this.getDimension().width, this.getDimension().height);
		area.intersect(new java.awt.geom.Area(clip));
		
		return this.applyRotation(area);
	}

}
