package graphics;

/**
 * A polygon is a Graphic that can form many shapes. To use, first set the
 * desired dimension of the polygon. Then add at least three points in order to
 * correctly make a polygon. This will then play connect the dots in the order
 * that the points were added. Then connects the last point back to the first,
 * forming a polygon.
 * <p />
 * For example, to create a triangle: <br/>
 *  graphics.Polygon triangle = new graphics.Polygon(); <br/>
 *	_triangle.setDimension(new java.awt.Dimension(30,30)); <br/>
 *	_triangle.addPoint(new Point(0,0)); <br/>
 *	_triangle.addPoint(new Point(30,0)); <br/>
 *	_triangle.addPoint(new Point(30,30)); <br/>
 * 
 * @author <a href="mailto:kozelsky@cse.buffalo.edu">Michael Kozelsky</a>
 * 
 * Created on: Jul 29, 2006 Polygon.java
 */
public class Polygon extends AbstractColorableGraphic {

	/** The underlying java.awt.Polygon **/
	private java.awt.Polygon _poly;
	
	/** The original (i.e. not rotated/resized) java.awt.Polygon **/
	private java.awt.Polygon _original;

	/**
	 * Creates a new instance of Polygon with no vertices
	 */
	public Polygon() {
		_poly = new java.awt.Polygon();
		_original = new java.awt.Polygon();
	}

	/**
	 * Add a point to the polygon
	 * 
	 * @param p
	 *            the point to add
	 */
	public void addPoint(java.awt.Point p) {
		_poly.addPoint(p.x, p.y);
		
		//everytime we add a point, redefine what the original/"correct" version of the polygon is
		this.defineOriginal();
		
		if (this.getContainer() != null)
			this.getContainer().repaint(this.getBounds());
	}

	/**
	 * removes the first occurrence of point p from the polygon
	 * 
	 * @param p
	 *            The Point to remove
	 */
	public void removePoint(java.awt.Point p) {
		// build arraylist
		java.util.ArrayList<java.awt.Point> vertices = new java.util.ArrayList<java.awt.Point>();
		for (int i = 0; i < _poly.npoints; i++) {
			vertices.add(new java.awt.Point(_poly.xpoints[i], _poly.ypoints[i]));
		}

		// remove p from arraylist
		vertices.remove(p);

		// build polygon from updated arraylist
		_poly.reset();
		for (java.awt.Point point : vertices) {
			_poly.addPoint(point.x, point.y);
		}

		this.getContainer().repaint(this.getBounds());
		
		this.defineOriginal();
	}

	/**
	 * Every time a point is added or removed, we should redefine what the 'original' polygon is.
	 */
	private void defineOriginal() {
		_original.reset();
		for (int i = 0; i < _poly.npoints; i++) {
			_original.addPoint(_poly.xpoints[i],_poly.ypoints[i]);
		}
	}

	/**
	 * This method uses the Graphics2D object to paint the Polygon onto the
	 * container.  The location and dimension parameters are not used because
	 * the polygon knows it's location and dimension 
	 * 
	 * @see graphics.IGraphic#actualPaint(java.awt.Graphics2D,
	 *      java.awt.Point, java.awt.Dimension)
	 * @param gs
	 *            The Graphics2D object to do the painting
	 * @param location
	 *            Does nothing
	 * @param dimension
	 *            Does Nothing
	 */
	public void actualPaint(java.awt.Graphics2D gs, java.awt.Point location, java.awt.Dimension dimension) {		
		gs.fillPolygon(_poly);
	}

	/**
	 * Sets the upper left hand corner of the polygon's bounding box.
	 * 
	 * @see graphics.ILocatable#setLocation(java.awt.Point)
	 * @param point
	 *            A java.awt.Point representing the new location of the Polygon
	 */
	@Override
	public void setLocation(java.awt.Point point) {
		java.awt.Rectangle oldBounds = this.getBounds();

		// find the offset of how much we need to move this
		int xOffset = point.x - this.getLocation().x;
		int yOffset = point.y - this.getLocation().y;

		// move the graphic
		_poly.translate(xOffset, yOffset);
		
		// update the internal reference of the location
		super.setLocation(_poly.getBounds().getLocation());

		if (this.getContainer() != null) {
			this.getContainer().repaint(this.getBounds().union(oldBounds));
		}
	}



	/**
	 * Resizes the Polygon to a specified dimension. 
	 * 
	 * @see graphics.ISizeable#setDimension(java.awt.Dimension)
	 * @param d
	 *            The dimension which the polygon should have
	 */
	@Override
	public void setDimension(java.awt.Dimension d) {
		super.setDimension(d);
		
		java.awt.Rectangle oldBounds = this.getBounds();
		
		// Save the loctation and rotation and reset them to 0.
		// this will need to be reapplied after resize
		java.awt.Point location = this.getLocation();
		this.setLocation(new java.awt.Point());
		Integer r = this.getRotation();
		this.setRotation(0);

		java.awt.Rectangle originalBounds = _original.getBounds();

		// the width of the new Dimension/current width = ratioX. Every x
		// location is multiplied by ratio
		double ratioX = (double) d.width / (double) originalBounds.width;

		// the height of the new Dimension/current length = ratioY. Every y
		// location is multiplied by ratio
		double ratioY = (double) d.height / (double) originalBounds.height;
		
		//reset _poly
		_poly.reset();
		
		// build up the newly resized polygon, assuming it is at the origin, will move back later
		for (int i = 0; i < _original.npoints; i++) {
			// the new point is the x-coordinate * ratioX and y-coordinate * ratioY
			_poly.addPoint(
					(int) (_original.xpoints[i] * ratioX),
					(int) (_original.ypoints[i] * ratioY));
		}

		// reset the location and rotation
		this.setLocation(location);
		this.setRotation(r);

		if (this.getContainer() != null)
			this.getContainer().repaint(this.getBounds().union(oldBounds));
		
	}
	
	/**
	 * Gets a java.awt.shape that represents this shape
	 * @see graphics.IGraphic#getShape()
	 * @return a java.awt.Polygon that represents this polygon
	 */
	public java.awt.Shape getShape() {
					
		return this.applyRotation(_poly);
	}

}