package physics;

/**
 * FrictionMotion.java
 * 
 * 
 * Created: Fri Nov 12 14:55:12 2004
 * 
 * @author <a href="mailto: "Sara M Haydanek</a>
 * @version An <code>Imotion</code> that mimics friction
 */

public class FrictionMotion extends IMotion {

	private double _dx, _dy;
	private int _speed;
	private CollisionObject _colOb;
	private MotionTimer _mt;
	private double _coefficient;
	// private double _DXaddingPixel, _DYaddingPixel;
	// private int _counter;
	private boolean _collided;// , _sub;
	private double _momentum, _ob2Momentum;
	private double _mPercent;

	/**
	 * Creates a new <code>FrictionMotion</code> instance. and set up default
	 * values
	 * 
	 * @param colOb
	 *            a <code>CollisionObject</code> value
	 */
	public FrictionMotion(CollisionObject colOb) {
		// _sub = false;
		// _DXaddingPixel = 0;
		// _DYaddingPixel = 0;
		_coefficient = .07;
		_speed = 100;
		_dx = 10;
		_dy = 0;
		_colOb = colOb;
		_mt = new MotionTimer(_speed, this);
		_mPercent = 0;
		_collided = false;
	}

	/**
	 * Sets the coefficient of friction
	 * 
	 * @param c
	 *            a <code>double</code>
	 */
	@Override
	public void setCoefficient(double c) {
		_coefficient = c;
	}

	/**
	 * Sets the velocity of the motion based on a dx, dy, and speed
	 * 
	 * @param dx
	 *            Change in x
	 * @param dy
	 *            Change in y
	 * @param speedTimer
	 *            delay
	 */
	@Override
	public void setVelocity(int dx, int dy, int speed) {
		// _sub = false;
		_dx = dx;
		_dy = dy;
		_speed = speed;
		_colOb.setLine(new LineFormula(_colOb.getShape().getLocation(), _dy
				/ _dx));
	}

	/**
	 * Set the dx direction
	 * 
	 * @param d -
	 *            the change in the x direction
	 */
	@Override
	public void setDX(double d) {
		_dx = d;
	}

	/**
	 * Set the dy direction
	 * 
	 * @param d
	 *            the change in the y direction
	 */
	@Override
	public void setDY(double d) {
		_dy = d;
	}

	/**
	 * Return the velocity of the object based on dy and dx
	 * 
	 * @return velocity
	 */
	@Override
	public double getVelocity() {
		double dy = _dy;
		double dx = _dx;
		if (dy < 1 && dy > -1) {
			dy = 0;
		} // end of if ()
		if (dx < 1 && dx > -1) {
			dx = 0;
		} // end of if ()
		return Math.sqrt((dy * dy) + (dx * dx));
	}

	/**
	 * Returns the speed of the motion
	 * 
	 * @return the timer delay value
	 */
	@Override
	public int getSpeed() {
		return _speed;
	}

	/**
	 * Returns dx of the velocity
	 * 
	 * @return change in x direction
	 */
	@Override
	public int getDX() {
		return new Double(_dx).intValue();
	}

	/**
	 * Returns dy of the velocity
	 * 
	 * @return change in y direction
	 */
	@Override
	public int getDY() {
		return new Double(_dy).intValue();
	}

	/**
	 * Moves the object at intervals decided by the timer and checks for
	 * collisions
	 */
	@Override
	public void timerReact() {
		if (_dx > .2 || _dx < -.2) {
			_dx = _dx - truncate(_dx * _coefficient);
		} // end of if ()

		if (_dy > .2 || _dy < -.2) {
			_dy = _dy - truncate(_dy * _coefficient);
		} // end of if ()
		this.setLocation(_dx, _dy);
		_colOb.detectCollision();
	}

	/**
	 * Sets the location of the object based on a double value for x and y
	 * 
	 * @param x -
	 *            x value
	 * @param y -
	 *            y value
	 */
	@Override
	public void setLocation(double x, double y) {
		_colOb.getShape().setCenterLocation(
				new java.awt.Point(_colOb.getShape().getCenterLocation().x
						+ new Double(x).intValue(), _colOb.getShape()
						.getCenterLocation().y
						+ new Double(y).intValue()));
	}

	/**
	 * Returns a truncated double truncated after 2 decimal places
	 * 
	 * @param db -
	 *            the double to truncate
	 * @return the truncated double
	 */
	public static double truncate(double db) {
		if (db > 0) {
			String s = (new Double(db)).toString();
			int index = s.indexOf('.');
			int end = index;
			if (s.length() > index + 1) {
				end = index + 1;
				if (s.length() > index + 2) {
					end = index + 2;
				} // end of if ()
			} // end of if ()
			if (end == index) {
				return db;
			} // end of if ()
			if (new Integer(s.charAt(0)).intValue() >= 1) {
				index = index - 1;
			} // end of if ()

			String s2 = s.substring(index, end + 1);
			return new Double(s2).doubleValue();
		}
		if (db < 0) {
			String s = (new Double(db)).toString();
			int index = s.indexOf('.');
			int end = index;
			if (s.length() > index + 1) {
				end = index + 1;
				if (s.length() > index + 2) {
					end = index + 2;
				} // end of if ()
			} // end of if ()
			if (end == index) {
				return db;
			} // end of if ()
			if (new Integer(s.charAt(1)).intValue() >= 1) {
				index = index - 1;
			} // end of if ()

			String s2 = s.substring(index, end + 1);
			return -(new Double(s2).doubleValue());
		} else {
			return db;
		} // end of else

	}

	/**
	 * Starts the MotionTimer for this motion
	 */
	@Override
	public void start() {
		_mt.start();
	}

	/**
	 * Stops the MotionTimer for this object
	 */
	@Override
	public void stop() {
		_mt.stop();
	}

	/**
	 * A bounce for this motion
	 */
	@Override
	public void bounce(CollisionObject col) {
		_collided = true;
		_momentum = _colOb.calculateMomentum();
		_ob2Momentum = col.calculateMomentum();
		if (col.getShape().getCenterLocation().x == _colOb.getShape()
				.getCenterLocation().x
				|| col.getShape().getCenterLocation().y == _colOb.getShape()
						.getCenterLocation().y) {
			this.switchMomentum(_ob2Momentum, col);
		} // end of if ()
		else {
			this.calculateMomentum(col);

			if (col.getShape().getCenterLocation().x == _colOb.getShape()
					.getCenterLocation().x
					|| col.getShape().getCenterLocation().y == _colOb
							.getShape().getCenterLocation().y
					|| this.getDX() == 0 || this.getDY() == 0) {
				// java.awt.Point p = _colOb.getShape().getLocation();
				int dx = this.getDX();
				int dy = this.getDY();
				dx = -dx;
				dy = -dy;
				this.setVelocity(dx, dy, this.getSpeed());
				_colOb.getShape().setCenterLocation(
						new java.awt.Point((_colOb.getShape()
								.getCenterLocation().x + dx), (_colOb
								.getShape().getCenterLocation().y + dy)));
			} else {
				/** change dx * */
				if ((this.getDX() < 0 && this.getDY() > 0)
						|| (this.getDX() > 0 && this.getDY() < 0)) {
					int dx = this.getDX();
					dx = -dx;
					this.setVelocity(dx, this.getDY(), this.getSpeed());
					_colOb.getShape()
							.setCenterLocation(
									new java.awt.Point((_colOb.getShape()
											.getCenterLocation().x + this
											.getDX()), (_colOb.getShape()
											.getCenterLocation().y + this
											.getDY())));
				} // end of if ()

				/** change dy * */
				else {

					if ((this.getDX() > 0 && this.getDY() > 0)
							|| (this.getDX() < 0 && this.getDY() < 0)) {
						int dy = this.getDY();
						dy = -dy;
						this.setVelocity(this.getDX(), dy, this.getSpeed());
						_colOb.getShape()
								.setCenterLocation(
										new java.awt.Point((_colOb.getShape()
												.getCenterLocation().x + this
												.getDX()), (_colOb.getShape()
												.getCenterLocation().y + this
												.getDY())));
					} // end of if ()
				} // end of else
			}// end of else

			this.stillCollide(col);
		} // end of else
	}

	/**
	 * Checks to see if the object that has this motion, and the param are still
	 * colliding, and if so moves them apart
	 * 
	 * @param col
	 *            a <code>CollisionObject</code> value
	 */
	public void stillCollide(CollisionObject col) {

		while (_colOb.getShape().intersects(col.getShape())) {
			if (this.getDX() > 0) {
				_colOb.getShape().setCenterLocation(
						new java.awt.Point(_colOb.getShape()
								.getCenterLocation().x - 2, _colOb.getShape()
								.getCenterLocation().y));
			} // end of if ()
			if (this.getDX() < 0) {
				_colOb.getShape().setCenterLocation(
						new java.awt.Point(_colOb.getShape()
								.getCenterLocation().x + 2, _colOb.getShape()
								.getCenterLocation().y));
			} // end of if ()
			if (this.getDY() < 0) {
				_colOb.getShape().setCenterLocation(
						new java.awt.Point(_colOb.getShape()
								.getCenterLocation().x, _colOb.getShape()
								.getCenterLocation().y + 2));
			} // end of if ()
			if (this.getDX() > 0) {
				_colOb.getShape().setCenterLocation(
						new java.awt.Point(_colOb.getShape()
								.getCenterLocation().x, _colOb.getShape()
								.getCenterLocation().y - 2));
			} // end of if ()
		} // end of while ()
		_collided = false;
	}

	/**
	 * Calculates the momentum of this object using the line formulas and the
	 * angle of collision
	 * 
	 * @param col
	 *            a <code>CollisionObject</code> value
	 */
	public void calculateMomentum(CollisionObject col) {
		FormulaSolver fs = new FormulaSolver(_colOb.getLine(), col.getLine());
		double x = fs.solveX();
		double y = fs.solveY();
		double xx2a = (x - col.getLine().getX0()) * (x - col.getLine().getX0());
		double xx2b = (x - _colOb.getLine().getX0())
				* (x - _colOb.getLine().getX0());
		double yy2a = (y - col.getLine().getY0()) * (y - col.getLine().getY0());
		double yy2b = (y - _colOb.getLine().getY0())
				* (y - _colOb.getLine().getY0());
		double distance1 = Math.sqrt(xx2a + yy2a);
		double distance2 = Math.sqrt(xx2b + yy2b);
		double xx2c = (col.getLine().getX0() - _colOb.getLine().getX0())
				* (col.getLine().getX0() - _colOb.getLine().getX0());
		double yy2c = (col.getLine().getY0() - _colOb.getLine().getY0())
				* (col.getLine().getY0() - _colOb.getLine().getY0());
		double distance3 = Math.sqrt(xx2c + yy2c);
		double numerator = -(distance3 * distance3) + (distance1 * distance1)
				+ (distance2 * distance2);
		double denominator = 2 * distance1 * distance2;
		double cosc = numerator / denominator;
		double theta = Math.acos(cosc);
		if (theta == 180) {
			this.switchMomentum(_ob2Momentum, col);
		} // end of if ()
		else {

			_mPercent = (90 - theta) / 90;
			double percentoff = _mPercent * _momentum;
			_momentum = _momentum - percentoff;
			col.getMotion().changeMomentum(percentoff);
		}
	}

	/**
	 * Switchs the momentum of the the object with this motion and col
	 * 
	 * @param m -
	 *            momentum
	 * @param col
	 *            a <code>CollisionObject</code> value
	 */
	@Override
	public void switchMomentum(double m, CollisionObject col) {
		double dx = _dx;
		double dy = _dy;
		if (m == 0) {
			_dx = 0;
			_dy = 0;
		} // end of if ()
		else {
			double newVelocity = m / _colOb.getMass();
			double extra = (Math.sqrt(((newVelocity * newVelocity)
					- (_dx * _dx) - (_dy * _dy)))) / 2;
			_dx = _dx + extra;
			_dy = _dy + extra;
		}
		if (col.getMotion().getDX() < 1 && col.getMotion().getDY() < 1
				&& !col.getMotion().getCollided()) {
			col.getMotion().setDX(dx);
			col.getMotion().setDY(dy);
			col.getMotion().setLocation(dx, dy);
		} // end of if ()

	}

	/**
	 * Changes the momentum of the object with this motion by subtracting the
	 * momentum it must give to the object with which it collided
	 * 
	 * @param percentoff
	 *            a <code>double</code> value
	 */
	@Override
	public void changeMomentum(double percentoff) {
		_momentum = _momentum + percentoff;
		double newVelocity = _momentum / _colOb.getMass();
		double newDY = Math.sqrt((newVelocity * newVelocity) - (_dx * _dx));
		this.setVelocity((int) _dx, (int) newDY, _speed);
	}

	/**
	 * Return whether it has collided
	 * 
	 * @return a <code>boolean</code> value
	 */
	@Override
	public boolean getCollided() {
		return _collided;
	}

	/**
	 * Sets the DX direction
	 */
	@Override
	public void setDX(int dx) {
		_dx = dx;
	}

	/**
	 * Sets the DY direction
	 */
	@Override
	public void setDY(int dy) {
		_dy = dy;
	}

}// FrictionMotion
