/*
This file is licensed under the terms of the Globus Toolkit Public
License, found at http://www.globus.org/toolkit/download/license.html.
*/
package org.globus.ogsa.impl.samples.notification;

import org.globus.ogsa.impl.ogsi.GridServiceImpl;
import org.globus.ogsa.GridConstants;
import org.globus.ogsa.GridContext;
import org.globus.ogsa.GridServiceException;
import org.globus.ogsa.NotificationProvider;
import org.globus.ogsa.ServiceProperties;
import org.globus.ogsa.client.managers.NotificationSinkManager;
import org.globus.ogsa.samples.notification.SinkListenerPortType;
import org.globus.ogsa.utils.AnyHelper;
import org.globus.ogsa.utils.GSIUtils;
import org.globus.ogsa.utils.SubscriptionHelper;
import org.globus.ogsa.wsdl.GSR;

import org.globus.axis.util.Util;

import org.gridforum.ogsi.ExtendedDateTimeType;
import org.gridforum.ogsi.ExtensibilityType;
import org.gridforum.ogsi.GridService;
import org.gridforum.ogsi.HandleType;
import org.gridforum.ogsi.LocatorType;
import org.gridforum.ogsi.NotificationSource;
import org.gridforum.ogsi.OGSIServiceGridLocator;
import org.gridforum.ogsi.holders.LocatorTypeHolder;
import org.gridforum.ogsi.holders.TerminationTimeTypeHolder;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import javax.xml.namespace.QName;
import javax.xml.rpc.Stub;

import java.net.URL;

import java.rmi.RemoteException;

import java.util.Calendar;
import java.util.Enumeration;
import java.util.Hashtable;


public class SinkListenerImpl extends GridServiceImpl {
    private OGSIServiceGridLocator serviceLocator =
        new OGSIServiceGridLocator();
    private HandleType handle;
    private Hashtable subscriptions = new Hashtable();
    private NotificationProvider sourceProvider;
    static Log logger = LogFactory.getLog(SinkListenerImpl.class.getName());

    public SinkListenerImpl() {
        super("Sink Listener Sample");
    }

    public void postCreate(GridContext context) throws GridServiceException {
	super.postCreate(context);

        // Set up Message Generator
        this.handle =
            new HandleType((String) getProperty(ServiceProperties.HANDLE));

	this.sourceProvider =
	    (NotificationProvider) getProperty(
                    ServiceProperties.NOTIFICATION_SOURCE
	);
	this.sourceProvider.addTopic("MessageGateway", 
				     new QName(GridConstants.XSD_NS, 
					       "anyType"));
    }

    public void addListener(
        String topic,
        HandleType sourceHandle
    ) throws RemoteException {
        if (
            this.subscriptions.get(topic + ":" + sourceHandle.getValue()) != null
        ) {
            throw new RemoteException(
                "SinkListener: subscription already exists"
            );
        }

        try {
            URL sourceURL = new URL(sourceHandle.toString());
            NotificationSource source =
                this.serviceLocator.getNotificationSourcePort(sourceURL);
            GSIUtils.setDefaultGSIProperties((Stub) source, sourceURL);

            Calendar timeout = Calendar.getInstance();
            timeout.add(Calendar.HOUR, 1);

            ExtendedDateTimeType dateTime = new ExtendedDateTimeType();
            dateTime.setValue(timeout);

            LocatorType locator = new LocatorType();
            locator.setHandle(new HandleType[] { this.handle });

            LocatorTypeHolder locatorHolder = new LocatorTypeHolder();
            source.subscribe(
                SubscriptionHelper.getNameExpression(new QName("", topic)),
                locator, dateTime, locatorHolder,
                new TerminationTimeTypeHolder()
            );

            GSR gsr = GSR.newInstance(locatorHolder.value);
            this.subscriptions.put(
                topic + ":" + sourceHandle.toString(), gsr.getEndpoint()
            );
        } catch (Exception e) {
            throw new GridServiceException("Failed to add listener", e);
        }
    }

    public void removeListener(
        String topic,
        HandleType sourceHandle
    ) throws RemoteException {
        try {
            String endpoint =
                (String) this.subscriptions.get(
                    topic + ":" + sourceHandle.toString()
                );
            URL subscriptionURL = new URL(endpoint);
            GridService subscription =
                this.serviceLocator.getGridServicePort(subscriptionURL);
            GSIUtils.setDefaultGSIProperties(
                (Stub) subscription, subscriptionURL
            );
            subscription.destroy();
            this.subscriptions.remove(topic + ":" + sourceHandle.getValue());
        } catch (Exception e) {
            throw new RemoteException(
                "Failed to remove listener: " + e.toString()
            );
        }
    }

    public void deliverNotification(ExtensibilityType message)
        throws RemoteException {
        try {
            this.sourceProvider.notify("MessageGateway", message.get_any());
        } catch (Exception e) {
            logger.error("Failed to notify: " + e.toString());
        }
    }

    public void preDestroy(GridContext context) throws GridServiceException {
	super.preDestroy(context);
	
        for (
            Enumeration enum = this.subscriptions.elements();
                enum.hasMoreElements();
        ) {
            try {
                String endpoint = (String) enum.nextElement();
                URL subscriptionURL = new URL(endpoint);
                GridService subscription =
                    this.serviceLocator.getGridServicePort(subscriptionURL);
                GSIUtils.setDefaultGSIProperties(
                    (Stub) subscription, subscriptionURL
                );
                subscription.destroy();
            } catch (Exception e) {
                logger.error("Failed to remove listener: " + e.toString());
            }
        }
    }
}
