package edu.buffalo.cse.jive.app.bst;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileReader;
import java.io.IOException;

import edu.buffalo.cse.jive.app.bst.BST.BSTException;

public class Controller {

	public static final class ControllerException extends Exception {

		private static final long serialVersionUID = 1588346935987794459L;

		ControllerException(final String message) {

			super(message);
		}

		ControllerException(final Throwable throwable) {

			super(throwable);
		}
	}

	private static final long ANIMATION_INTERVAL = 500;
	private final Model model;
	private final View view;
	private Thread workerThread;

	public Controller(final boolean hasView) {

		model = new Model();
		view = hasView ? new View(this) : null;
	}

	public void closeView() throws ControllerException {

		if (view == null) {
			return;
		}
		try {
			// System.err.println("sleeping");
			Thread.sleep(2 * ANIMATION_INTERVAL);
		} catch (InterruptedException e) {
			e.printStackTrace();
		}
		try {
			view.close();
		} catch (final Exception e) {
			throw new ControllerException(e);
		}
	}

	public Model getModel() {

		return model;
	}

	private final class ModelLoader implements Runnable {

		private final File file;

		private ModelLoader(final File file) {
			this.file = file;
		}

		private int parseIntArg(final String input) {

			final int spaceIndex = input.indexOf(" ");
			if (spaceIndex != (input.length())) {
				final String arg = input.substring(spaceIndex + 1);
				return Integer.parseInt(arg);
			}
			throw new IllegalArgumentException();
		}

		@Override
		public void run() {

			try {
				final BufferedReader in = new BufferedReader(new FileReader(
						file));
				String line = null;
				while ((view == null || !view.done())
						&& (line = in.readLine()) != null) {
					line = line.trim().toLowerCase();
					try {
						if (line.startsWith("#")) {
							continue;
						} else if (line.equals("clear")) {
							model.clear();
						} else if (line.startsWith("insert ")) {
							model.insertNode(parseIntArg(line));
						} else if (line.startsWith("remove ")) {
							model.removeNode(parseIntArg(line));
						} else if (line.startsWith("sleep ")) {
							Thread.sleep(parseIntArg(line));
						} else {
							model.insertNode(Integer.parseInt(line));
						}
						if (view != null) {
							updateView();
							Thread.sleep(ANIMATION_INTERVAL);
						}
					} catch (final NumberFormatException nfe) {
						nfe.printStackTrace();
					} catch (final BSTException be) {
						be.printStackTrace();
					}
				}
			} catch (final IOException ioe) {
				ioe.printStackTrace();
			} catch (final InterruptedException e) {
				if (!view.done()) {
					System.err.println("InterruptedException::done");
					e.printStackTrace();
				}
			}
		}
	}

	public void loadModel(final File file) {

		// spawn a background worker thread to load the tree
		// new ModelLoader(file).run();
		workerThread = new Thread(new ModelLoader(file));
		workerThread.setDaemon(true);
		workerThread.start();
	}

	public void showView() throws ControllerException {

		if (view == null) {
			return;
		}
		try {
			view.show();
		} catch (final Exception e) {
			throw new ControllerException(e);
		}
	}

	public void updateView() {

		if (view == null) {
			return;
		}
		if (view.done()) {
			System.err.println("updateView::done");
			Thread.currentThread().interrupt();
		}
		view.updateUI();
	}

	public void waitForWorker() throws ControllerException {

		try {
			workerThread.join();
		} catch (final InterruptedException e) {
			throw new ControllerException(e);
		}
	}
}