package edu.buffalo.cse.jive.app.bst.animation;

import java.awt.BasicStroke;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.RenderingHints;
import java.awt.geom.Ellipse2D;

import javax.swing.JComponent;

import edu.buffalo.cse.jive.app.bst.BST;

public class TreePanel extends JComponent {

  private static final long serialVersionUID = 1667058772086193507L;

  private final int deltaY = 60;
  private final BasicStroke edgeStroke;
  private final int gutter = 5;
  private Image image;
  private int imageWidth, imageHeight;
  private final Font nodeFont;
  private final BasicStroke nodeStroke;
  private final RenderingHints renderingHints;

  TreePanel() {

    // rendering hints
    renderingHints = new RenderingHints(RenderingHints.KEY_ANTIALIASING,
        RenderingHints.VALUE_ANTIALIAS_ON);
    renderingHints.put(RenderingHints.KEY_RENDERING, RenderingHints.VALUE_RENDER_QUALITY);

    // node font
    nodeFont = new Font(Font.SANS_SERIF, Font.BOLD, 12);

    // strokes
    edgeStroke = new BasicStroke(0.75f);
    nodeStroke = new BasicStroke(0.5f);
  }

  public void clear() {

    newImage();
    getGraphics().drawImage(getImage(), 0, 0, this);
  }

  protected void drawBF(final Graphics g, final int subtreeWidth, final BST tree, final int x,
      final int y) {

    final String text = Integer.toString(tree.data());
    final Graphics2D g2d = (Graphics2D) g;

    g2d.setRenderingHints(renderingHints);

    // font stuff
    g2d.setFont(nodeFont);
    final FontMetrics fm = g2d.getFontMetrics();
    final int sw = fm.stringWidth(text);
    final int sh = fm.getHeight();

    // draw node contour (for help only)
    // g2d.setColor(new Color(255, 248, 192));
    // g2d.setStroke(new BasicStroke(0.5f, BasicStroke.CAP_BUTT, BasicStroke.JOIN_BEVEL, 0,
    // new float[] { 5, 2 }, 0));
    // g2d.fill(new Rectangle2D.Double(x - subtreeWidth / 2, y, subtreeWidth, deltaY));
    // g2d.setColor(new Color(255, 0, 0));
    // g2d.draw(new Rectangle2D.Double(x - subtreeWidth / 2, y, subtreeWidth, deltaY));

    // draw node edges first
    g2d.setColor(Color.BLACK);
    g2d.setStroke(edgeStroke);
    if (tree.left() != null) {
      g2d.drawLine(x - sw / 2, y + gutter + sh, x - subtreeWidth / 2, y + deltaY);
    }
    if (tree.right() != null) {
      g2d.drawLine(x + sw / 2, y + gutter + sh, x + subtreeWidth / 2, y + deltaY);
    }

    // draw node
    final Ellipse2D node = new Ellipse2D.Double(x - gutter - sw / 2, y, 2 * gutter + sw, 2 * gutter
        + sh);
    g2d.setColor(Color.LIGHT_GRAY);
    g2d.fill(node);
    g2d.setColor(Color.BLACK);
    g2d.setStroke(nodeStroke);
    g2d.draw(node);

    // draw node text
    g2d.setColor(Color.BLACK);
    g2d.drawString(text, x - sw / 2, y + gutter + sh - fm.getDescent());

    // draw left subtree
    if (tree.left() != null) {
      drawBF(g, subtreeWidth / 2, tree.left(), x - subtreeWidth / 2, y + deltaY);
    }

    // draw right subtree
    if (tree.right() != null) {
      drawBF(g, subtreeWidth / 2, tree.right(), x + subtreeWidth / 2, y + deltaY);
    }
  }

  public void drawTree(final BST tree) {

    final Dimension d = getBounds().getSize();
    if (getImage() == null || getImageWidth() != d.width || getImageHeight() != d.height) {
      newImage();
    }
    if (tree != null) {
      drawBF(getImage().getGraphics(), getImageWidth() / 2, tree, getImageWidth() / 2, 10);
    }
    getGraphics().drawImage(getImage(), 0, 0, this);
  }

  protected Image getImage() {

    return image;
  }

  protected int getImageHeight() {

    return imageHeight;
  }

  protected int getImageWidth() {

    return imageWidth;
  }

  protected void newImage() {

    final Dimension d = getBounds().getSize();
    image = createImage(d.width, d.height);
    imageWidth = d.width;
    imageHeight = d.width;
    final Graphics gr = image.getGraphics();
    gr.setColor(getBackground());
    gr.fillRect(0, 0, imageWidth, imageHeight);
  }

  @Override
  public void paint(final Graphics g) {

    if (image == null) {
      newImage();
    }
    g.drawImage(image, 0, 0, this);
  }
}