package edu.buffalo.cse.jive.app.concurrency;

/* From http://java.sun.com/docs/books/tutorial/index.html */
/*
 * Copyright (c) 2006 Sun Microsystems, Inc. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * -Redistribution of source code must retain the above copyright notice, this
 *  list of conditions and the following disclaimer.
 *
 * -Redistribution in binary form must reproduce the above copyright notice,
 *  this list of conditions and the following disclaimer in the documentation
 *  and/or other materials provided with the distribution.
 *
 * Neither the name of Sun Microsystems, Inc. or the names of contributors may
 * be used to endorse or promote products derived from this software without
 * specific prior written permission.
 *
 * This software is provided "AS IS," without a warranty of any kind. ALL
 * EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND WARRANTIES, INCLUDING
 * ANY IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE
 * OR NON-INFRINGEMENT, ARE HEREBY EXCLUDED. SUN MIDROSYSTEMS, INC. ("SUN")
 * AND ITS LICENSORS SHALL NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY LICENSEE
 * AS A RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
 * DERIVATIVES. IN NO EVENT WILL SUN OR ITS LICENSORS BE LIABLE FOR ANY LOST
 * REVENUE, PROFIT OR DATA, OR FOR DIRECT, INDIRECT, SPECIAL, CONSEQUENTIAL,
 * INCIDENTAL OR PUNITIVE DAMAGES, HOWEVER CAUSED AND REGARDLESS OF THE THEORY
 * OF LIABILITY, ARISING OUT OF THE USE OF OR INABILITY TO USE THIS SOFTWARE,
 * EVEN IF SUN HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
 *
 * You acknowledge that this software is not designed, licensed or intended
 * for use in the design, construction, operation or maintenance of any
 * nuclear facility.
 */
import java.awt.Color;
import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import javax.swing.BorderFactory;
import javax.swing.ImageIcon;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JSlider;
import javax.swing.SwingConstants;
import javax.swing.SwingUtilities;
import javax.swing.WindowConstants;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

public class DiningPhilosophers extends JFrame implements ActionListener, ChangeListener {

  private static final long serialVersionUID = 399550908526070662L;

  private JButton stopStartButton = new JButton("start");

  static final int NUMPHILS = 5;
  static final int HUNGRYDUKE = 0;
  static final int RIGHTSPOONDUKE = 1;
  static final int BOTHSPOONSDUKE = 2;
  private static final double MARGIN = 20.0f;

  // delays can go from 0 to 10,000 milliseconds, initial value is 500
  private JSlider grabDelaySlider = new JSlider(JSlider.HORIZONTAL, 0, 100, 20);
  private JLabel label = new JLabel(" 1000 milliseconds");
  private JPanel philosopherArea;
  private int width = 0;
  private int height = 0;
  private Philosopher[] philosophers = new Philosopher[NUMPHILS];
  public ImageIcon[] imgs = new ImageIcon[3];
  int grabDelay = 1250;
  Chopstick[] chopsticks = new Chopstick[NUMPHILS];
  String[] names = { "Arisduktle", "Dukrates", "Pythagoduke", "Duko", "Dukimedes" };

  private Dimension preferredSize;

  public void init() {

    imgs[HUNGRYDUKE] = createImageIcon("hungryduke.gif");
    imgs[RIGHTSPOONDUKE] = createImageIcon("rightspoonduke.gif");
    imgs[BOTHSPOONSDUKE] = createImageIcon("bothspoonsduke.gif");

    width = imgs[HUNGRYDUKE].getIconWidth() + (int) (MARGIN * 2.0);
    height = imgs[HUNGRYDUKE].getIconHeight() + (int) (MARGIN * 2.0);

    GridBagLayout gridBag = new GridBagLayout();
    GridBagConstraints c = new GridBagConstraints();

    JPanel contentPane = new JPanel();
    contentPane.setLayout(gridBag);

    philosopherArea = new JPanel(null);
    philosopherArea.setBackground(Color.white);
    philosopherArea.setBorder(BorderFactory.createCompoundBorder(
        BorderFactory.createLoweredBevelBorder(), BorderFactory.createEmptyBorder(15, 15, 15, 15)));
    preferredSize = createPhilosophersAndChopsticks();
    philosopherArea.setPreferredSize(preferredSize);

    c.fill = GridBagConstraints.BOTH;
    c.weighty = 1.0;
    c.gridwidth = GridBagConstraints.REMAINDER; // end row
    gridBag.setConstraints(philosopherArea, c);
    contentPane.add(philosopherArea);

    c.fill = GridBagConstraints.HORIZONTAL;
    c.weightx = 1.0;
    c.weighty = 0.0;
    gridBag.setConstraints(stopStartButton, c);
    contentPane.add(stopStartButton);

    c.gridwidth = GridBagConstraints.RELATIVE; // don't end row
    c.weightx = 1.0;
    c.weighty = 0.0;
    gridBag.setConstraints(grabDelaySlider, c);
    contentPane.add(grabDelaySlider);

    c.weightx = 0.0;
    c.gridwidth = GridBagConstraints.REMAINDER; // end row
    gridBag.setConstraints(label, c);
    contentPane.add(label);
    contentPane.setBorder(BorderFactory.createEmptyBorder(15, 15, 15, 15));
    setContentPane(contentPane);

    stopStartButton.addActionListener(this);
    grabDelaySlider.addChangeListener(this);

  }

  /** Returns an ImageIcon, or null if the path was invalid. */
  protected static ImageIcon createImageIcon(String path) {

    java.net.URL imgURL = DiningPhilosophers.class.getResource(path);
    if (imgURL != null) {
      return new ImageIcon(imgURL);
    }
    else {
      System.err.println("Couldn't find file: " + path);
      return null;
    }
  }

  @Override
  public void actionPerformed(ActionEvent e) {

    if (stopStartButton.getText().equals("stop/reset")) {
      stopPhilosophers();
      stopStartButton.setText("start");
    }
    else if (stopStartButton.getText().equals("start")) {
      startPhilosophers();
      stopStartButton.setText("stop/reset");
    }
  }

  @Override
  public void stateChanged(ChangeEvent e) {

    JSlider source = (JSlider) e.getSource();
    grabDelay = source.getValue() * 100;
    label.setText(String.valueOf(grabDelay + " milliseconds"));
  }

  public void startPhilosophers() {

    for (int i = 0; i < NUMPHILS; i++)
      philosophers[i].philThread.start();
  }

  public void stopPhilosophers() {

    for (int i = 0; i < NUMPHILS; i++)
      philosophers[i].philThread.interrupt();
  }

  public Dimension createPhilosophersAndChopsticks() {

    double x, y;
    double radius = 100.0;
    double centerAdjX = 200.0;
    double centerAdjY = 100.0;
    double radians;

    Dimension preferredSize = new Dimension(0, 0);

    /*
     * for a straight line y = MARGIN;
     */
    for (int i = 0; i < NUMPHILS; i++)
      chopsticks[i] = new Chopstick();

    for (int i = 0; i < NUMPHILS; i++) {
      /*
       * for a straight line x = i * spacing;
       */
      radians = i * (2.0 * Math.PI / NUMPHILS);
      x = Math.sin(radians) * radius + centerAdjX;
      y = Math.cos(radians) * radius + centerAdjY;
      philosophers[i] = new Philosopher(this, i, imgs[HUNGRYDUKE]);
      philosophers[i].setBounds((int) x, (int) y, width, height);
      philosopherArea.add(philosophers[i]);
      if ((int) x > preferredSize.width)
        preferredSize.width = (int) x;
      if ((int) y > preferredSize.height)
        preferredSize.height = (int) y;
    }

    preferredSize.width += width;
    preferredSize.height += height;
    return preferredSize;
  }

  public static void main(String[] args) {

    final DiningPhilosophers demo = new DiningPhilosophers();
    demo.setPreferredSize(new Dimension(540, 460));
    demo.setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);
    demo.init();
    SwingUtilities.invokeLater(new Runnable() {

      @Override
      public void run() {

        demo.pack();
        demo.setVisible(true);
      }
    });
  }
}

/*
 * This class requires no changes from the 1.0 version. It's kept here so the rest of the example
 * can compile.
 */

class Philosopher extends JLabel implements Runnable {

  private static final long serialVersionUID = -2517780617599319361L;
  private Chopstick leftStick, rightStick;
  private DiningPhilosophers parent;
  private int position;
  Thread philThread = null;

  public Philosopher(DiningPhilosophers parent, int position, ImageIcon img) {

    super(parent.names[position], img, SwingConstants.CENTER);

    this.parent = parent;
    this.position = position;

    setVerticalTextPosition(SwingConstants.BOTTOM);
    setHorizontalTextPosition(SwingConstants.CENTER);

    // identify the chopsticks to my right and left
    this.rightStick = parent.chopsticks[position];
    if (position == 0) {
      this.leftStick = parent.chopsticks[DiningPhilosophers.NUMPHILS - 1];
    }
    else {
      this.leftStick = parent.chopsticks[position - 1];
    }

    philThread = new Thread(this);
  }

  @Override
  public void run() {

    try {
      while (true) {
        Thread.sleep((int) (Math.random() * parent.grabDelay * 2));
        rightStick.grab();

        // grabbed right
        setIcon(parent.imgs[DiningPhilosophers.RIGHTSPOONDUKE]);
        Thread.sleep((int) (Math.random() * parent.grabDelay * 2));
        leftStick.grab();

        // grabbed left
        setIcon(parent.imgs[DiningPhilosophers.BOTHSPOONSDUKE]);
        setText("mmmm!");
        Thread.sleep((int) (Math.random() * parent.grabDelay));
        setText("     ");

        // release left
        leftStick.release();
        setIcon(parent.imgs[DiningPhilosophers.RIGHTSPOONDUKE]);
        Thread.sleep((int) (Math.random() * parent.grabDelay * 2));

        // release right
        rightStick.release();
        setIcon(parent.imgs[DiningPhilosophers.HUNGRYDUKE]);
        Thread.sleep((int) (Math.random() * parent.grabDelay * 2));
      }
    }
    catch (java.lang.InterruptedException e) {
    }
    leftStick.releaseIfMine();
    rightStick.releaseIfMine();
    setIcon(parent.imgs[DiningPhilosophers.HUNGRYDUKE]);
    setText(parent.names[position]);
    philThread = new Thread(this);
  }
}

class Chopstick {

  Thread holder = null;

  public synchronized void grab() throws InterruptedException {

    while (holder != null)
      wait();
    holder = Thread.currentThread();
  }

  public synchronized void release() {

    holder = null;
    notify();
  }

  public synchronized void releaseIfMine() {

    if (holder == Thread.currentThread())
      holder = null;
    notify();
  }
}