package edu.buffalo.cse.jive.app.dblp.model;

import java.io.File;
import java.io.IOException;

import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import com.sun.org.apache.xerces.internal.parsers.SAXParser;

import edu.buffalo.cse.jive.app.dblp.sampler.XMLPersister;
import edu.buffalo.cse.jive.app.dblp.sampler.XMLSampler;
import edu.buffalo.cse.jive.app.dblp.sampler.XMLWriterFactory;
import edu.buffalo.cse.jive.app.dblp.sql.SQLGenerator;
import edu.buffalo.cse.jive.app.dblp.sql.SQLPersister;
import edu.buffalo.cse.jive.app.dblp.sql.SQLWriterFactory;

public class Processor {

  private static final String XML = ".xml";
  private static final String SAMPLE_FOLDER_FMT = "%ssql-%s/";
  public static final String[] SAMPLE_IDS = { "01", "02", "03", "04", "05" };
  private static final String SAMPLE_NAME_FMT = "%s%s-sample-%s%s";
  private static final double[] SAMPLE_RATES = { 0.005, 0.01, 0.02, 0.04, 0.08 };

  public boolean generateOutput;
  private long globalTimeStart;
  private final String xmlFolderName;
  private final String dblpName;
  private final String sqlFolderName;

  public Processor(final boolean generateOutput, final String xmlFolderName, final String dblpName,
      final String sqlFolderName) {

    this.xmlFolderName = xmlFolderName;
    this.sqlFolderName = sqlFolderName;
    this.dblpName = dblpName;
    this.generateOutput = generateOutput;
  }

  private void message(final String message) {

    if (generateOutput) {
      System.err.println(message);
    }
  }

  private void printEntityStats(final WriterFactory factory) throws IOException {

    if (factory instanceof SQLWriterFactory) {
      printStats((SQLWriterFactory) factory);
    }
    else if (factory instanceof XMLWriterFactory) {
      printStats((XMLWriterFactory) factory);
    }
  }

  private void printStats(final SQLWriterFactory factory) throws IOException {

    if (!generateOutput) {
      return;
    }
    final long globalTime = System.nanoTime();
    final Float elapsed = new Float(globalTime - globalTimeStart) / 1000000000.0f;
    final Float entities = new Float(factory.defaultPersister().entityCounter()) / 1000.0f;
    final Float throughput = entities / elapsed * 1000.0f;
    System.out.println(String.format("elapsed time...........: %2.0f  sec", elapsed));
    System.out.println(String.format("total entities.........: %5.0fK entities", entities));
    System.out.println(String.format("average throughput.....: %5.0f  entities/sec", throughput));
  }

  public void printStats(final String section) {

    if (!generateOutput) {
      return;
    }
    final long globalTime = System.nanoTime();
    final Float elapsed = new Float(globalTime - globalTimeStart) / 1000000000.0f;
    System.out.println(String.format("Section................: %s", section));
    System.out.println(String.format("elapsed time...........: %2.0f  sec", elapsed));
  }

  private void printStats(final XMLWriterFactory factory) throws IOException {

    if (!generateOutput) {
      return;
    }
    final long globalTime = System.nanoTime();
    final Float elapsed = new Float(globalTime - globalTimeStart) / 1000000000.0f;
    final Float entities = new Float(factory.defaultPersister().entityCounter()) / 1000.0f;
    final Float throughput = entities / elapsed * 1000.0f;
    System.out.println(String.format("elapsed time...........: %2.0f  sec", elapsed));
    System.out.println(String.format("total entities.........: %5.0fK entities", entities));
    System.out.println(String.format("average throughput.....: %5.0f  entities/sec", throughput));
  }

  /**
   * Generic XML processing template, for both sampling and sql generation.
   */
  public void processXML(final WriterFactory factory, final ContentHandler handler,
      final String parserTarget) throws IOException, SAXException {

    final SAXParser parser = new SAXParser();
    parser.setContentHandler(handler);
    final Persister persister = factory.defaultPersister();
    persister.prepare();
    printStats("initialization");
    parser.parse(parserTarget);
    printStats("parsing");
    persister.flush();
    printEntityStats(factory);
  }

  /**
   * Processes a single DBLP file, generating the corresponding SQL files to populate a RDBMS.
   */
  public void sqlProcessSample(final int sampleId) throws IOException, SAXException {

    time();
    final String parserTarget = String.format(SAMPLE_NAME_FMT, xmlFolderName, dblpName,
        SAMPLE_IDS[sampleId], XML);
    final String sqlTarget = String.format(SAMPLE_FOLDER_FMT, sqlFolderName, SAMPLE_IDS[sampleId]);
    message("\nProcessing " + parserTarget + " and writing to " + sqlFolderName);
    final File sqlFolder = new File(sqlTarget);
    if (!sqlFolder.exists()) {
      message("creating folder " + sqlFolder.getPath());
      sqlFolder.mkdir();
    }
    else {
      message("deleting contents of folder " + sqlFolder.getPath());
      for (final File file : sqlFolder.listFiles()) {
        if (file.canWrite()) {
          file.delete();
        }
      }
    }
    final WriterFactory factory = new SQLWriterFactory(sqlTarget, false);
    final ContentHandler handler = new SQLGenerator((SQLPersister) factory.defaultPersister());
    processXML(factory, handler, parserTarget);
  }

  /**
   * Samples an input DBLP file using a fixed sampling rate.
   */
  public void xmlSample(final int sampleId) throws IOException, SAXException {

    time();
    final String parserTarget = xmlFolderName + dblpName + XML;
    final WriterFactory factory = new XMLWriterFactory(String.format(SAMPLE_NAME_FMT, xmlFolderName, 
      dblpName, SAMPLE_IDS[sampleId], XML));
    final ContentHandler handler = new XMLSampler((XMLPersister) factory.defaultPersister(),
        SAMPLE_RATES[sampleId]);
    message("\nProcessing " + parserTarget + " using a sample rate of " + SAMPLE_RATES[sampleId]
        + " and writing to " + String.format(SAMPLE_NAME_FMT, xmlFolderName, dblpName, SAMPLE_IDS[sampleId], XML));
    processXML(factory, handler, parserTarget);
  }

  public void time() {

    globalTimeStart = System.nanoTime();
  }

  public int sampleCount() {

    return SAMPLE_IDS.length;
  }
}