package edu.buffalo.cse.jive.app.linkedlist;

public abstract class ListFactory {

  public static <T extends Comparable<T>> JiveList<T> createList() {

    return new LinkedList<T>();
  }

  public static JiveIntList createIntList() {

    return new IntLinkedList();
  }

  /**
   * List of integers.
   */
  public interface JiveIntList {

    public boolean contains(int element);

    public void insert(int element);

    public boolean remove(int element);

    public int size();
  }

  /**
   * Generic list of comparable types.
   */
  public interface JiveList<T extends Comparable<T>> {

    public boolean contains(T element);

    public void insert(T element);

    public boolean remove(T element);

    public int size();
  }

  /**
   * Generic linked list implementation that maintains list elements in sorted order and uses
   * iteration in both insert and delete. Duplicates are supported.
   */
  private static class LinkedList<T extends Comparable<T>> implements JiveList<T> {

    private ListNode head;
    private int size;

    public LinkedList() {

      size = 0;
      head = null;
    }

    @Override
    public boolean contains(T element) {

      ListNode ptr = head();
      while (ptr != null && ptr.data().compareTo(element) < 0) {
        ptr = ptr.next();
      }
      return (ptr != null && ptr.data().compareTo(element) == 0);
    }

    public ListNode head() {

      return head;
    }

    @Override
    public void insert(T element) {

      ListNode prev = null;
      ListNode ptr = head();
      while (ptr != null && ptr.data().compareTo(element) < 0) {
        prev = ptr;
        ptr = ptr.next();
      }
      if (prev == null) {
        head = new ListNode(element, head());
      }
      else {
        prev.next = new ListNode(element, ptr);
      }
      size++;
    }

    @Override
    public boolean remove(T element) {

      ListNode prev = null;
      ListNode ptr = head();
      while (ptr != null && ptr.data().compareTo(element) < 0) {
        prev = ptr;
        ptr = ptr.next();
      }
      if (ptr != null && ptr.data().compareTo(element) == 0) {
        if (prev != null) {
          prev.next = ptr.next();
        }
        else {
          head = head().next();
        }
        size--;
        return true;
      }
      return false;
    }

    @Override
    public int size() {

      return size;
    }

    @Override
    public String toString() {

      StringBuffer result = new StringBuffer("[");
      if (head != null) {
        result.append(head.toString());
      }
      return result.append("]").toString();
    }

    /**
     * This class cannot be made static because of its dependence on the generic type T.
     */
    private class ListNode {

      private T data;
      private ListNode next;

      public ListNode(T data, ListNode next) {

        this.data = data;
        this.next = next;
      }

      public T data() {

        return data;
      }

      public ListNode next() {

        return next;
      }

      @Override
      public String toString() {

        StringBuffer result = new StringBuffer("");
        ListNode ptr = this;
        while (ptr != null) {
          result.append(ptr.data());
          if (ptr.next() != null) {
            result.append(",");
          }
          ptr = ptr.next();
        }
        return result.toString();
      }
    }
  }

  /**
   * Linked list implementation that maintains integers elements in sorted order and uses recursion
   * in both insert and delete. Duplicates are supported.
   */
  private static class IntLinkedList implements JiveIntList {

    private int size;
    private ListNode head;

    public IntLinkedList() {

      size = 0;
      head = null;
    }

    private ListNode findParent(int element) {

      // special case: head is null
      if (head == null) {
        return null;
      }
      // special case: head's data is larger than element
      if (head.data() > element) {
        return null;
      }
      // special case: head's data is equal to element
      if (head.data() == element) {
        return head;
      }
      // general case: find the last node n' such that n'.data < element
      ListNode prev = head.find(element);
      // the found node is either null or its data is equal to or larger than element
      ListNode found = prev.next();
      // the only case that is an actual match
      if (found != null && found.data() == element) {
        return prev;
      }
      return null;
    }

    @Override
    public boolean contains(int element) {

      return findParent(element) != null;
    }

    @Override
    public void insert(int element) {

      if (head == null || head.data() >= element) {
        head = new ListNode(element, head);
      }
      else {
        ListNode found = head.find(element);
        found.linkTo(new ListNode(element, found.next()));
      }
      size++;
    }

    @Override
    public boolean remove(int element) {

      // special case: empty list
      if (head == null) {
        return false;
      }
      // special case: head's data is equal to element
      if (head.data() == element) {
        head = head.next();
        size--;
        return true;
      }
      // general case: find the parent of the node that contains element
      ListNode found = findParent(element);
      // not found
      if (found == null) {
        return false;
      }
      // found-- link the parent to the grandchild
      found.linkTo(found.next().next());
      size--;
      return true;
    }

    @Override
    public int size() {

      return size;
    }

    @Override
    public String toString() {

      StringBuffer result = new StringBuffer("[");
      if (head != null) {
        result.append(head.toString());
      }
      return result.append("]").toString();
    }

    /**
     * Private static implementation of the list's node.
     * 
     * @author Demian Lessa
     * 
     */
    private static class ListNode {

      private int data;
      private ListNode next;

      public ListNode(int data, ListNode next) {

        this.data = data;
        this.next = next;
      }

      public int data() {

        return data;
      }

      /**
       * Returns: (1) the last ListNode smaller than element, if one exists; or (2) this ListNode.
       */
      public ListNode find(int element) {

        if (data() >= element) {
          return this;
        }
        ListNode node = (ListNode) next();
        if (node == null || node.data() >= element) {
          return this;
        }
        return node.find(element);
      }

      public ListNode next() {

        return next;
      }

      public void linkTo(ListNode other) {

        next = other;
      }

      @Override
      public String toString() {

        StringBuffer result = new StringBuffer("");
        ListNode ptr = this;
        while (ptr != null) {
          result.append(ptr.data());
          if (ptr.next() != null) {
            result.append(",");
          }
          ptr = ptr.next();
        }
        return result.toString();
      }
    }
  }
}