package edu.buffalo.cse.jive.app.linkedlist.test;

import java.util.ArrayList;
import java.util.List;

import edu.buffalo.cse.jive.app.linkedlist.ListFactory;
import edu.buffalo.cse.jive.app.linkedlist.ListFactory.JiveList;

import static edu.buffalo.cse.jive.app.linkedlist.test.ListTestSuite.LIST_MAX;

import junit.framework.TestCase;

public class ListTest extends TestCase {

  private JiveList<Integer> myList;

  protected void setUp() throws Exception {

    super.setUp();
    myList = ListFactory.createList();
  }

  protected void tearDown() throws Exception {

    myList = null;
    super.tearDown();
  }

  /**
   * Test for the empty list (corner case).
   */
  public void testEmpty() {

    assertNotNull(myList);
    assertEquals(0, myList.size());
    assertFalse(myList.contains(0));
    assertFalse(myList.contains(1));
    assertFalse(myList.remove(1));
    assertFalse(myList.remove(0));
  }

  /**
   * Test for the unary list (corner case).
   */
  public void testUnary() {

    assertNotNull(myList);
    assertEquals(0, myList.size());
    myList.insert(0);
    assertEquals(1, myList.size());
    assertTrue(myList.contains(0));
    assertFalse(myList.contains(1));
    assertFalse(myList.remove(1));
    assertTrue(myList.remove(0));
    assertFalse(myList.remove(0));
    assertEquals(0, myList.size());
    assertFalse(myList.contains(0));
    assertFalse(myList.contains(1));
  }

  /**
   * Inserts several random elements, possibly duplicating some of them.
   */
  public void testInsert() {

    assertNotNull(myList);
    assertEquals(0, myList.size());
    for (int i = LIST_MAX; i > 0; i--) {
      int element = new Double(LIST_MAX * Math.random()).intValue();
      myList.insert(element);
      assertEquals(LIST_MAX - i + 1, myList.size());
      assertTrue(myList.contains(element));
    }
    assertEquals(LIST_MAX, myList.size());
  }

  /**
   * Inserts several random elements, forcing duplication of half of them.
   */
  public void testInsertDuplicates() {

    assertNotNull(myList);
    assertEquals(0, myList.size());
    for (int i = LIST_MAX; i > 0; i--) {
      int element = new Double(LIST_MAX * Math.random()).intValue();
      myList.insert(element);
      if (i % 2 == 0) {
        myList.insert(element);
      }
      assertEquals(LIST_MAX - i + 1 + ((LIST_MAX - i + 2) / 2), myList.size());
      assertTrue(myList.contains(element));
    }
    assertEquals(3 * LIST_MAX / 2, myList.size());
  }

  /**
   * Inserts several elements, caches the insertions into a local list known to work, then removes
   * all cached elements. Note that we cannot test contains right after the removal since the list
   * may contain duplicates. We do check that each removal was successful (returned true) and that
   * the sizes match after each removal.
   */
  public void testRemove() {

    List<Integer> mirror = new ArrayList<Integer>();
    assertNotNull(myList);
    assertEquals(0, myList.size());
    for (int i = LIST_MAX; i > 0; i--) {
      int element = new Double(LIST_MAX * Math.random()).intValue();
      myList.insert(element);
      assertEquals(LIST_MAX - i + 1, myList.size());
      assertTrue(myList.contains(element));
      mirror.add(element);
    }
    int size = mirror.size();
    for (int element : mirror) {
      assertTrue(myList.contains(element));
      assertTrue(myList.remove(element));
      size--;
      assertEquals(size, myList.size());
    }
    assertEquals(0, myList.size());
    for (int element : mirror) {
      assertFalse(myList.contains(element));
    }
  }

  /**
   * Inserts and removes several elements.
   */
  public void testRemoveZigZag() {

    assertNotNull(myList);
    assertEquals(0, myList.size());
    for (int i = LIST_MAX; i > 0; i--) {
      int element = new Double(LIST_MAX * Math.random()).intValue();
      myList.insert(element);
      assertTrue(myList.contains(element));
      assertEquals(1, myList.size());
      assertTrue(myList.remove(element));
      assertFalse(myList.contains(element));
      assertEquals(0, myList.size());
    }
  }
}