package edu.buffalo.cse.jive.app.sorting;

import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;

public abstract class SortAlgorithm<T extends Comparable<T>> {

  public interface AnimationListener extends SortAlgorithmListener {

    void showComparing(AlgorithmId algorithm, int index1, int index2);

    void showSwapping(AlgorithmId algorithm, int index1, int index2);
  }

  public interface SortAlgorithmListener {
  }

  public interface TerminationListener extends SortAlgorithmListener {

    void done(AlgorithmId algorithm);
  }

  private final static int COMPARE_DELAY = 20;
  private final static int SWAP_DELAY = 80;

  private final AlgorithmId algoId;
  private boolean done;
  private final boolean isAnimated;
  private final List<SortAlgorithmListener> listeners = new CopyOnWriteArrayList<SortAlgorithmListener>();
  private final SortableCollection<T> values;
  private final Thread worker;

  SortAlgorithm(final AlgorithmId algoId, final SortableCollection<T> values,
      final SortAlgorithmFactory factory, final boolean isAnimated) {

    this.done = false;
    this.values = values;
    this.algoId = algoId;
    this.isAnimated = isAnimated;
    this.worker = factory.createSorterThread(this);
  }

  public void addListener(final SortAlgorithmListener listener) {

    listeners.add(listener);
  }

  protected int compare(final int index1, final int index2) {

    try {
      for (final SortAlgorithmListener l : listeners) {
        if (l instanceof AnimationListener) {
          ((AnimationListener) l).showComparing(algoId, index1, index2);
        }
      }
      if (isAnimated) {
        Thread.sleep(COMPARE_DELAY);
      }
    }
    catch (final InterruptedException e) {
    }
    return getData().compare(index1, index2);
  }

  void done() {

    this.done = true;
    for (final SortAlgorithmListener l : listeners) {
      if (l instanceof TerminationListener) {
        ((TerminationListener) l).done(algoId);
      }
    }
//    try {
//      Thread.sleep(4*COMPARE_DELAY);
//    }
//    catch (InterruptedException e) {
//      // TODO Auto-generated catch block
//      e.printStackTrace();
//    }
  }

  public SortableCollection<T> getData() {

    return values;
  }

  public AlgorithmId getId() {

    return this.algoId;
  }

  public boolean isDone() {

    return this.done;
  }

  protected abstract void sort();

  public void start() {

    worker.start();
  }

  protected void swap(final int index1, final int index2) {

    try {
      getData().swap(index1, index2);
      for (final SortAlgorithmListener l : listeners) {
        if (l instanceof AnimationListener) {
          ((AnimationListener) l).showSwapping(algoId, index1, index2);
        }
      }
      if (isAnimated) {
        Thread.sleep(SWAP_DELAY);
      }
    }
    catch (final InterruptedException e) {
    }
  }

  @Override
  public String toString() {

    return algoId.toString();
  }
}