package edu.buffalo.cse.jive.app.sorting;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CountDownLatch;

import edu.buffalo.cse.jive.app.sorting.SortAlgorithm.AnimationListener;
import edu.buffalo.cse.jive.app.sorting.SortAlgorithm.TerminationListener;
import edu.buffalo.cse.jive.data.DataFactory;

public class SortApplication implements TerminationListener {

  private final CountDownLatch endSignal;
  private int runningCount = 5;
  private final Map<AlgorithmId, SortAlgorithm<Double>> sorters = new HashMap<AlgorithmId, SortAlgorithm<Double>>();
  private final CountDownLatch startSignal;
  private long startTime;

  public SortApplication(final int dataSize) {

    this(dataSize, null, false);
  }

  public SortApplication(final int dataSize, final boolean isAnimated) {

    this(dataSize, null, isAnimated);
  }

  public SortApplication(final int dataSize, final AnimationListener listener,
      final boolean isAnimated) {

    startSignal = new CountDownLatch(1);
    endSignal = new CountDownLatch(runningCount);
    final List<Double> commonData = DataFactory.createDoubleList(dataSize, 5, 95);
    for (final AlgorithmId algoId : AlgorithmId.values()) {
      final SortAlgorithm<Double> algorithm = SortAlgorithmFactory.createAlgorithm(algoId,
          SortableCollection.create(commonData), startSignal, endSignal, isAnimated);
      // listen to termination
      algorithm.addListener(this);
      // listen to animation
      if (listener != null) {
        algorithm.addListener(listener);
      }
      algorithm.start();
      sorters.put(algoId, algorithm);
    }
  }

  @Override
  public synchronized void done(final AlgorithmId algorithm) {

    runningCount--;
    System.out.println(String.format("%s completed in %6.3f sec.", algorithm,
        1.0 * (System.nanoTime() - startTime) / (1000 * 1000 * 1000)));
    System.out.println(runningCount + " sorter threads still running.");
  }

  public SortAlgorithm<Double> getAlgorithm(final AlgorithmId algoId) {

    return sorters.get(algoId);
  }

  public void start() {

    startTime = System.nanoTime();
    // release all threads
    startSignal.countDown();
    try {
      // wait for all threads to complete
      endSignal.await();
    }
    catch (final InterruptedException e) {
      // TODO Auto-generated catch block
      e.printStackTrace();
    }
  }
}