package edu.buffalo.cse.jive.app.sorting;

import java.util.Collection;
import java.util.List;
import java.util.Vector;

/**
 * SortableCollection is an abstract data type that encapsulates a collection and exposes two simple
 * methods for you to use in your sort algorithms: compare, to compare two elements of the
 * collection at the given indices, and swap, to exchange the elements of the collection at the
 * given indices. Other methods are used for information purposes, such as getComparisons and
 * getSwaps, that keep track of the number of such operations performed by the algorithm.
 * 
 * @updated 09/22/2011
 * @date 12/01/2008
 * @author Demian Lessa <dlessa@cse.buffalo.edu>
 */
public class SortableCollection<T extends Comparable<T>> {

  private int dataSwaps = 0;
  private int dataComparisons = 0;
  private Vector<T> data = new Vector<T>();

  /**
   * Create a SortableCollection object by passing a regular Java collection such as a Vector. This
   * class encapsulates the data and exposes the necessary operations for you to implement your sort
   * algorithms.
   * 
   * @param collection
   *          represents the data in this collection
   */
  private SortableCollection(final Collection<T> collection) {

    for (T item : collection) {
      data.add(item);
    }
  }

  public static <E extends Comparable<E>> SortableCollection<E> create(final List<E> source) {

    return new SortableCollection<E>(source);
  }

  /**
   * Retrieves the element at the given index in the collection.
   * 
   * @precondition 0 <= index < getSize()
   * @param index
   *          is the position of the element to retrieve
   * @return element at the given index
   */
  public T get(int index) {

    return data.get(index);
  }

  /**
   * Retrieves the number of comparisons made between elements of the collection so far.
   * 
   * @return total number of calls to compare
   */
  public int getComparisons() {

    return dataComparisons;
  }

  /**
   * Retrieves the number of elements in the collection.
   * 
   * @return number of elements in the collection
   */
  public int size() {

    return data.size();
  }

  /**
   * Retrieves the number of swaps made between elements of the collection so far.
   * 
   * @return total number of calls to swap
   */
  public int getSwaps() {

    return dataSwaps;
  }

  /**
   * Compares two elements of the underlying collection.
   * 
   * @precondition 0 <= index1 < getSize()
   * @precondition 0 <= index2 < getSize()
   * @param index1
   *          is the index of the left element to compare
   * @param index1
   *          is the index of the right element to compare
   * @return result of compareTo applied to elements in the given collection positions
   */
  int compare(int index1, int index2) {

    dataComparisons++;
    return data.get(index1).compareTo(data.get(index2));
  }

  /**
   * Swaps two elements of the underlying collection.
   * 
   * @precondition 0 <= index1 < getSize()
   * @precondition 0 <= index2 < getSize()
   * @param index1
   *          is the index of the left element to swap
   * @param index1
   *          is the index of the right element to swap
   */
  void swap(int index1, int index2) {

    dataSwaps++;
    T temp = data.get(index1);
    data.set(index1, data.get(index2));
    data.set(index2, temp);
  }
}