package edu.buffalo.cse.jive.app.sorting.animation;

import java.awt.*;

import java.awt.geom.Rectangle2D;

import javax.swing.JComponent;
import javax.swing.SwingUtilities;

import edu.buffalo.cse.jive.app.sorting.AlgorithmId;
import edu.buffalo.cse.jive.app.sorting.SortAlgorithm;
import edu.buffalo.cse.jive.app.sorting.SortAlgorithm.AnimationListener;

/**
 * UI client of a sort algorithm.
 */
public class SortPanel extends JComponent implements AnimationListener {

  private static final long serialVersionUID = -5504122670835424164L;

  private final Integer BAR_SEP = 2;
  private final Integer MARK_SEP = 5;
  private final Integer LEFT_SEP = 35;
  private final Integer TEXT_SEP = 5;
  private final Integer TEXT_HEIGHT = 12;
  private final Integer HEADER_HEIGHT = 20;
  private final Integer FOOTER_HEIGHT = (TEXT_SEP + TEXT_HEIGHT) * 3;
  private int marked1;
  private int marked2;
  private Boolean isComparing = false;
  private Boolean isSwapping = false;
  private final SortAlgorithm<Double> algorithm;

  SortPanel(final SortAlgorithm<Double> algorithm) {

    super();
    this.algorithm = algorithm;
    setSize(new Dimension(250, 280));
    algorithm.addListener(this);
  }

  void start() {

    algorithm.start();
  }

  private Rectangle2D getClientRect() {

    return new Rectangle2D.Double(0, 0, getWidth(), getHeight());
  }

  private int getBarHeight() {

    return (getHeight() - HEADER_HEIGHT - FOOTER_HEIGHT - BAR_SEP
        * (algorithm.getData().size() - 1))
        / algorithm.getData().size();
  }

  private Rectangle2D getBar(final int index) {

    return new Rectangle2D.Double(LEFT_SEP, HEADER_HEIGHT + getBarHeight() * index + BAR_SEP
        * (index - 1), algorithm.getData().get(index) * (getWidth() - LEFT_SEP - MARK_SEP) / 100,
        getBarHeight());
  }

  private Rectangle2D getComparisonBar(final int index) {

    return new Rectangle2D.Double(MARK_SEP, HEADER_HEIGHT + getBarHeight() * index + BAR_SEP
        * (index - 1), LEFT_SEP - 2 * MARK_SEP, getBarHeight());
  }

  private Rectangle2D getConnectionBar(final int index1, final int index2) {

    int index = Math.min(index1, index2);
    return new Rectangle2D.Double(MARK_SEP, HEADER_HEIGHT + getBarHeight() * index + BAR_SEP
        * (index - 1), getBarHeight(), (Math.abs(index1 - index2) + 1) * getBarHeight() + BAR_SEP
        * Math.abs(index1 - index2));
  }

  @Override
  public void paintComponent(final Graphics g) {

    if (algorithm.getData() == null) {
      return;
    }
    defaultPaint(g);
    if (isComparing) {
      showComparing(g);
    }
    else if (isSwapping) {
      showSwapping(g);
    }
    Graphics2D g2 = (Graphics2D) g;
    g2.setPaint(Color.BLACK);
    g2.drawString(algorithm.toString(), 10, getHeight() - 3 * TEXT_HEIGHT - TEXT_SEP);
    g2.drawString("comparisons: " + algorithm.getData().getComparisons(), 10, getHeight() - 2
        * TEXT_HEIGHT - TEXT_SEP);
    g2.drawString("swaps: " + algorithm.getData().getSwaps(), 10, getHeight() - TEXT_HEIGHT
        - TEXT_SEP);
    g2.drawString("total: "
        + (algorithm.getData().getComparisons() + algorithm.getData().getSwaps()), 10, getHeight()
        - TEXT_SEP);
  }

  private void defaultPaint(final Graphics g) {

    Graphics2D g2 = (Graphics2D) g;
    g2.setPaint(Color.GRAY);
    g2.fill(getClientRect());
    for (int i = 0; i < algorithm.getData().size(); i++) {
      g2.setPaint(Color.BLACK);
      g2.fill(getBar(i));
    }
  }

  private void showComparing(final Graphics g) {

    Graphics2D g2 = (Graphics2D) g;
    g2.setPaint(Color.RED);
    g2.fill(getComparisonBar(marked1));
    g2.fill(getComparisonBar(marked2));
    g2.fill(getConnectionBar(marked1, marked2));
    g2.fill(getBar(marked1));
    g2.fill(getBar(marked2));
  }

  private void showSwapping(final Graphics g) {

    Graphics2D g2 = (Graphics2D) g;
    g2.setPaint(Color.BLUE);
    g2.fill(getComparisonBar(marked1));
    g2.fill(getComparisonBar(marked2));
    g2.fill(getConnectionBar(marked1, marked2));
    g2.fill(getBar(marked1));
    g2.fill(getBar(marked2));
  }

  @Override
  public void showComparing(final AlgorithmId algoId, final int index1, final int index2) {

    isSwapping = false;
    isComparing = true;
    marked1 = index1;
    marked2 = index2;
    safeRepaint();
  }

  @Override
  public void showSwapping(final AlgorithmId algoId, final int index1, final int index2) {

    isSwapping = true;
    isComparing = false;
    marked1 = index1;
    marked2 = index2;
    safeRepaint();
  }

  private void safeRepaint() {

    SwingUtilities.invokeLater(new Runnable() {

      @Override
      public void run() {

        repaint();
      }
    });
  }
}