;;; rlog-tools.el --- RCS log formatting tools

;; Copyright (C) 1993--2004 Hans Chalupsky

;; Author:   Hans Chalupsky <hans@cs.buffalo.edu>
;; Created:  Sept. 25, 91
;; Modified: Sept. 17, 93
;; Version:  $Id: rlog-tools.el,v 1.4 2004/08/26 23:24:24 snwiz Exp $


;; This program is free software; you may redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; A copy of the GNU General Public License can be obtained from this
;; program's author (send electronic mail to hans@cs.buffalo.edu) or from
;; the Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA
;; 02139, USA.


;;; Commentary:

;; Set of tools to format logfiles for RCS/CVS controlled files that were
;; created with the `rlog' command, for example, the output generated from
;; a command like
;;
;;  % rlog system.lisp,v myload.lisp,v sneps/ds/path.lisp,v
;;
;; can be converted automatically into a prettier format, similar to the one
;; we used to create by hand to document the changes to SNePS.

;; The basic idea is to take the ouput produced by the command above (in
;; the case of SNePS the list of files would be all 180 or so RCS controlled
;; files) clean it from any unnecessary information, add the proper filename
;; to every single revision of a file, sort all the change entries, and, if
;; desired, add for every such entry the differences to the previous revision
;; (generated by the `rcsdiff' command).

;; The initial rlog can be created with the GNU Emacs-19 dired by recursively
;; editing the whole source repository with the `-R' option, marking all files
;; with interesting extensions, e.g., `\.lisp,v$', `\.atn$', `\.dat$', 
;; `\.sneps$', `\.snlog$', etc., and then using the `&' or `!' command to
;; perfom `rlog' on all these files. A time restricted `rlog' can also be
;; created. For example, to log all changes after February 16, 92 one would
;; supply the following shell command in dired:
;;
;;      rlog -d1992/02/16\< *
;;
;; After the rlog has completed simply select the buffer containing the log
;; and do
;;
;;      M-x rlt-make-log-from-rlog-buffer
;;
;; which will do all the necessary cleaning, sorting and so on. After that
;; the modified buffer has to be saved to disk.   If diffs are wanted
;; too one has to be a bit patient, because rcsdiff has to be run on every
;; single revision. The resulting logfile - even though created almost fully
;; automatically - looks pretty much like the logfiles we previously had to
;; generate by hand.

;; Of course, the quality of the final log is only as good as the individual
;; log messages that were supplied on check-in of the modified files.

;; Restrictions:

;; This package assumes rlog output format of RCS version 5 or later. It was
;; only tested with GNU Emacs 19.19. In different Emacs implementations your
;; mileage may vary.

;; Installation:

;; Simply byte-compile and load this file.


;;; Code:

(defvar rlt-rlog-header "^RCS file: *\\([^ \t\n]*\\)"
  "Regexp to find the header part of the output from `rlog <file>'.")

(defvar rlt-rlog-rcs-file 1
  "RCS file field number in `rlt-rlog-header'.")

(defvar rlt-rlog-separator
  "^=============================================================================$"
  "Regexp that matches the last line of the rlog output.")

(defvar rlt-revision-header
  (concat "^revision +\\([0-9.]+\\).*\n"
          "date: *\\([12]?[90]?[7-9]./../.. ..:..:..\\); +"
	  "author: *\\([^;]*\\); +"
	  "state: *\\([^;]*\\);.*$")
  "Regexp that matches the header of a single revision in an rlog output.")

;; Field numbers for the individual components
;; in a revision header:
(defvar rlt-revision-number 1)
(defvar rlt-revision-date 2)
(defvar rlt-revision-author 3)
(defvar rlt-revision-state 4)

(defvar rlt-revision-separator "^----------------------------$"
  "Regexp that matches the separators between individual revisions")
(defvar rlt-revision-separator-string "----------------------------"
  "Separator string to replace some other separators")

(defun rlt-get-field (field &optional string)
  "Returns the FIELD's arg of the most recent regexp match in the current
buffer. If STRING is supplied it uses it as a source instead of the buffer."
  (cond (string (substring string
			   (match-beginning field)
			   (match-end field)))
	(t (buffer-substring (match-beginning field)
			     (match-end field)))))

(defun rlt-convert-rlog-to-log-buffer ()
  (interactive)
  (let ((case-fold-search nil)
	rlog-end rcs-file)
    (save-excursion
      (save-restriction
	(goto-char (point-min))
	(setq rlog-end (point-min))
	;; Search to the beginning of the first revision description
	;; in the rlog output. This way we automatically skip stuff
	;; with "selected revisions: 0":
	(while (re-search-forward rlt-revision-header nil t)
	  (re-search-backward rlt-rlog-header)
	  ;; remember the working file name:
	  (setq rcs-file (substring (rlt-get-field rlt-rlog-rcs-file) 0 -2))
	  ;; search to the start of the first revision description:
	  (re-search-forward rlt-revision-header)
          ;; goto its beginning:
	  (goto-char (match-beginning 0))
	  ;; delete all junk between the end of the previous rlog output
	  ;; and the start of the first revision description of this one:
	  (delete-region rlog-end (point))
	  ;; narrow to the rlog output for the current file:
	  (narrow-to-region (point) (re-search-forward rlt-rlog-separator))
	  (goto-char (point-min))
	  ;; change all revision headers into the desired format:
	  (while (re-search-forward rlt-revision-header nil t)
	    (replace-match
	     (format "File: %s (\\%s \\%s \\%s)"
		     rcs-file
		     rlt-revision-number
		     rlt-revision-date
		     rlt-revision-author)
	     t))
	  ;; replace the rlog output end separator with a revision separator:
	  (re-search-forward rlt-rlog-separator)
	  (replace-match rlt-revision-separator-string t)
	  (widen)
	  ;; remember the end of this converted rlog output
	  (setq rlog-end (1+ (point))))
	;; delete all remaining junk in the buffer
	(delete-region rlog-end (point-max))
	(message "Finished conversion to log format.")))))


;; Now that we have a log buffer generated from raw rlog output,
;; we can now sort the log buffer and insert DIFFS if necessary:

(defvar rlt-log-entry-header
  (concat "^File: +\\([^ ]+\\) +("
	  "\\([0-9.]+\\) +"
	  "\\([12][90][7-9]./../.. ..:..:..\\) +"
	  ".*$")
  "Regexp that matches the header of a log file entry.")

;; Field indices for the various items in a log entry
(defvar rlt-log-entry-file 1)
(defvar rlt-log-entry-revision 2)
(defvar rlt-log-entry-time 3)

;this does not work because log-entries can get pretty large in which case
;the regexp matching screws up and the sorting with it, sigh....
;(defun rlt-sort-log-buffer (reverse)
;  "Sorts a log buffer. Most recent changes will be at the top of the file
;if REVERSE is t."
;  (interactive "P")
;  (sort-regexp-fields reverse rlt-log-entry (format "\\%d" rlt-log-entry-time)
;		      (point-min) (point-max)))

(defun rlt-sort-log-buffer (reverse)
  "Sorts a log buffer. Most recent changes at the top if REVERSE is t."
  (interactive "P")
  (save-excursion
    (goto-char (point-min))
    (let ((log-buffer (current-buffer))
	  (date-buffer (get-buffer-create " *rlt-log-dates*"))
	  (sort-buffer (get-buffer-create " *rlt-log-sorted*"))
	  (case-fold-search nil)
	  log-entry-start)
      (set-buffer date-buffer)
      (erase-buffer)
      (set-buffer sort-buffer)
      (erase-buffer)
      (set-buffer log-buffer)
      ;; For every log entry insert its date and buffer position
      ;; into the date buffer whose lines will be sorted later:
      (while (re-search-forward rlt-log-entry-header nil t)
	(setq date-line (format "%s %s\n"
				(rlt-get-field rlt-log-entry-time)
				(match-beginning 0)))
	(set-buffer date-buffer)
	(insert date-line)
	(set-buffer log-buffer))
      ;; Find lower right corner of rectangle for `sort-columns':
      (set-buffer date-buffer)
      (goto-char (point-max))
      (search-backward " ")
      ;; Sort the date buffer:
      (sort-columns reverse (point-min) (point))
      (goto-char (point-min))
      ;; Now map over date buffer line by line, find the corresponding
      ;; entry in the log buffer and append it to the sort buffer:
      (while (< (point) (point-max))
	(forward-char 20)
	(setq log-entry-start (read date-buffer))
	(forward-char 1)
	(save-excursion
	  (set-buffer log-buffer)
	  (goto-char log-entry-start)
	  (re-search-forward rlt-revision-separator nil)
	  (forward-char 1)
	  (append-to-buffer sort-buffer log-entry-start (point))))
      (set-buffer log-buffer)
      (if (not (= (point-max)
		  (save-excursion (set-buffer sort-buffer) (point-max))))
	  (error "rlt-sort-log-buffer: Error during sorting")
	;; If everything went ok, replace the log buffer with
	;; the sorted contents:
	(erase-buffer)
	(insert-buffer sort-buffer)
	(message "Finished sorting.")))))

(defvar rlt-rcs-revision-number
   "^\\(\\([0-9]+\\)\\(\\.[0-9]+\\.[0-9]+\\)*\\)\\.\\([0-9]+\\)$"
   "Regexp that matches an RCS revision number")
(defvar rlt-rcs-revision-branch 1
  "Field index of the branch component of an RCS revision number.")

(defun rlt-previous-revision (revision)
  "Takes an rcs REVISON number (a string) and computes the previous revison
from it. If REVISION is the beginning of a branch it just returns it 
unchanged. This function is probably not as general as it should be but for
now it works."
  (string-match rlt-rcs-revision-number revision)
  (let* ((branch (rlt-get-field rlt-rcs-revision-branch revision))
	 (minor-number
	  (rlt-get-field (- (/ (length (match-data)) 2) 1) revision))
	 (minor-number-as-number
	  (car (read-from-string minor-number))))
    (cond (;; dumb check whether we are at the start of a branch
	   ;; if yes just return unchanged revision
	   (<= minor-number-as-number 1) revision)
	  (t (format "%s.%s" branch (1- minor-number-as-number))))))

(defun rlt-insert-diffs (root-directory)
  "Maps over a log file of change descriptions and inserts the differences
to the previous revision of the file into its change entry. If the filenames
in the log entries are relative on has to supply their ROOT-DIRECTORY, 
otherwise rcsdiff will not work properly."
  (interactive "DRoot directory: ")
  (save-excursion
    (goto-char (point-min))
    (let (file revision previous-revision diff-start)
      (while (re-search-forward rlt-log-entry-header (point-max) t)
	(setq file (format "%s,v" (rlt-get-field rlt-log-entry-file)))
	(setq revision (rlt-get-field rlt-log-entry-revision))
	(setq previous-revision (rlt-previous-revision revision))
	(re-search-forward rlt-revision-separator)
	(goto-char (match-beginning 0))
	(cond (;; initial revision?
	       (equal revision previous-revision))
	      (t (insert "\n")
		 (setq diff-start (point))
		 (message "rcsdiff -r%s -r%s %s"
			  previous-revision revision file)
		 (call-process "rcsdiff" nil t t
			       (concat "-r" previous-revision)
			       (concat "-r" revision)
			       "-c3"
			       (concat root-directory file))
		 (insert "\n")
		 (goto-char diff-start)
		 (re-search-forward "^diff .*+-r[^ ]+ .*-r[^ ]+.*")
		 (delete-region diff-start (match-beginning 0))
		 (goto-char diff-start)
		 (end-of-line)
		 (insert ":")))))))

(defun rlt-make-log-from-rlog-buffer ()
  "This one does it all: It converts the current rlog buffer into log
format, sorts the entries and inserts diffs if desired."
  (interactive)
  (let* ((reverse (y-or-n-p "Last changes first? "))
	 (diffs (y-or-n-p "Include diffs too? ")))
    (rlt-convert-rlog-to-log-buffer)
    (rlt-sort-log-buffer reverse)
    (if diffs (call-interactively 'rlt-insert-diffs))
    (message "All done, now save this buffer!")))


;;; rlog-tools.el ends here
