;;; -*- Mode: Lisp; Syntax: Common-Lisp; Package: GINSENG; Base: 10 -*-

;; Copyright (C) 1984--2004
;; Research Foundation of State University of New York

;; Version: $Id: flavors.lisp,v 1.5 2004/08/26 23:25:03 snwiz Exp $

;; This file is part of SNePS.

;; SNePS is free software; you may redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; SNePS is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with SNePS; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA, or to
;; Dr. Stuart C. Shapiro, Department of Computer Science and Engineering,
;; University at Buffalo, The State University of New York, 
;; 201 Bell Hall, Buffalo, NY 14260, USA

(in-package :ginseng)


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                            ;;;
;;;  NODE -- Flavor & Methods  ;;;
;;;                            ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


(defflavor node
	((arcs nil)		; A list of all arc instances of this Node.
	xpos			; The X-Coordinate of this Node.
	ypos			; The Y-Coordinate of this Node.
	the-node		; The Sneps accesible object.
	(typeof-object 'node)
	(radius *min-node-radius*) ; The radius of the node.
	(label nil))		; The Node label.
	()
  :settable-instance-variables
  :gettable-instance-variables
  :initable-instance-variables
  (:required-init-keywords :xpos :ypos))



(defmethod (:draw-self node) (&optional (window *window*)(alu tv:alu-ior)) 
  (let ((xpos (- (send self :xpos) (first *viewport-position*)))
	(ypos (- (send self :ypos) (second *viewport-position*)))
	(radius (send self :radius)))
    (cond ((sneps:isbase.n the-node)
	   (send window :draw-unfilled-ellipse xpos ypos (+ radius 4) 10 alu)
	   (let ((from-x (- xpos (// radius 2) 7))	; Label's X coordinate
		 (from-y (+ 4 ypos)))		; Label's Y coordinate
	     ;; Label the node
	     (send window :draw-string label
		   from-x from-y (1+ from-x) from-y
		   nil (list ':fix ':roman *font-size*) alu))
	   ;; Make mouse-sensitive box around node.
	   (send window :primitive-item :new-type self
		 (- xpos radius 3) (- ypos 10) (+ xpos radius 4) (+ ypos 11)))
	  (t
	   (send window :draw-unfilled-circle xpos ypos radius alu)
	   (let ((from-x (- xpos (// radius 2) 3))	; Label's X coordinate
		 (from-y (+ 11 (- ypos (// radius 2)))))	;Label's Y coordinate
	     ;; Label the node
	     (send window :draw-string label
		   from-x from-y (1+ from-x) from-y
		   nil (list ':fix ':roman *font-size*) alu))
	   ;; Make mouse-sensitive box around node.
	   (send window :primitive-item :new-type self
		 (- xpos radius) (- ypos radius) (+ xpos radius 2) (+ ypos radius 2)))))
  (unless (member self *displayed-nodes*)
    (push self *displayed-nodes*)))

(defmethod (node :draw-moving-node) (&optional (window *window*))
  (if (sneps:isbase.n the-node)
      (send window  :draw-unfilled-ellipse xpos ypos (1+ radius) 12 tv:alu-ior 3)
      (send window :draw-unfilled-circle xpos ypos radius tv:alu-ior 3)))


(defmethod (:init node)  (&rest ignore)
  (send self :reset-radius)			;
  (unless (member self *displayed-nodes*)
    (push self *displayed-nodes*)))


(defmethod (:reset-radius node)()
  (send self :set-radius
	(if label
	    ;; Establish radius with respect to the size of the label
	    (if (sneps::isbase.n the-node)
		(max *min-node-radius*
		     (+ *node-margin-size*
		        (round (* 1.2
				  (+ 3 (position *font-size* *avail-font-sizes*))
				  (// (scl:string-length (send self :label)) 2)))))
		(max *min-node-radius*
		     (+ *node-margin-size*
			(ceiling (* (+ 1.1 (position *font-size* *avail-font-sizes*))
				    (// (scl:string-length (send self :label)) 2))))))
	    *min-node-radius*)))


(defmethod (:add-arc node) (arc)
     (push arc arcs))


(defmethod (:in-view node)()
  "Is this node in the viewport?"
  (let ((x-pos (send self :xpos))
	(y-pos (send self :ypos))
	(radius (+ (send self :radius) 4)))
    (and (< (first *viewport-position*)  (- x-pos radius))
	 (< (second *viewport-position*) (- y-pos radius))
	 (> (third *viewport-position*)  (+ x-pos radius))
	 (> (fourth *viewport-position*)  (+ y-pos radius)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                           ;;;
;;;  ARC -- Flavor & Methods  ;;;
;;;                           ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defflavor arc
     ((arc-label nil)		; The Arc Label.
      (bent nil)		; Boolean on whether the arc is drawn as a spline.
      (middle-point nil)        ; An (x,y) along the bent arc.
      (typeof-object 'arc)
     node1			; The Source Node.
     node2)			; The Target Node of the arc.
     ()
  (:settable-instance-variables arc-label middle-point bent)
  (:initable-instance-variables node1 node2 arc-label)
  (:gettable-instance-variables typeof-object node1 node2 arc-label middle-point bent))



(defmethod (:init arc) (&rest ignore)
  "Initializes the arc data structure."
     (send node1 :add-arc self)
     (send node2 :add-arc self))


(Defmethod (:draw-self arc) (&optional (window *window*)(alu tv:alu-ior))
  "Determines whether the arc is bent and refers arc to the appropriate
      method for display."
  (if bent
      (send self :draw-bent-self window alu)
      (send self :draw-straight-self window alu)))


(Defmethod (:draw-bent-self arc) (window alu)
  "Displays an arc that has been bent."
  (multiple-value-bind (from-x from-y)
      ;; Establish "from" endpoint of arc to the mid-point.
      (calculate-node-edges
	(send node1 :radius)(send node1 :xpos) (send node1 :ypos)
	(sneps:isbase.n (send node1 :the-node))
	5 (first middle-point)(second middle-point)
	(sneps:isbase.n (send node2 :the-node))); Doesn't matter
    (multiple-value-bind (dummy-x dummy-y to-x to-y)
	;; Establish "to" endpoint of arc from the mid-point.
	(calculate-node-edges
	  5 (first middle-point)(second middle-point)
	  (sneps:isbase.n (send node1 :the-node)); Doesn't matter
	  (send node2 :radius)(send node2 :xpos) (send node2 :ypos) 
	  (sneps:isbase.n (send node2 :the-node)))	;
    
      (multiple-value-bind (label-from-x label-from-y)
	  ;; Establish starting point of the label.
	  (calculate-arc-label-position
	    (first middle-point) (second middle-point) to-x to-y arc-label)
	;; Draw the arc.
	(send window :draw-bent-arrow
	      from-x from-y (first middle-point)(second middle-point) to-x to-y)
	;; Label the arc.
	(send window :draw-string
	      (string arc-label)
	      (- label-from-x (left *viewport-position*))
	      (- label-from-y (second *viewport-position*))
	      (- to-x (first *viewport-position*))
	      (- to-y (second *viewport-position*))
	      nil (list ':fix ':roman *font-size*) alu)
	;; Set up a new mouse-sensitive area on the screen for this arc.
 	(send window :primitive-item :new-type self
	      (- (min label-from-x to-x)
		 (left *viewport-position*))
	      (- (min to-y label-from-y)
		 (top *viewport-position*)
		 10)
	      (- (max label-from-x to-x)
		 (left *viewport-position*))
	      (- (max to-y label-from-y)
		 (top *viewport-position*)))))))


(Defmethod (:draw-straight-self arc) (window alu)
  "Displays a straight arc."
  (multiple-value-bind (from-x from-y to-x to-y)
      ;; Establish endpoints of arc
      (calculate-node-edges
	(send node1 :radius)(send node1 :xpos) (send node1 :ypos)
	(sneps:isbase.n (send node1 :the-node))
	(send node2 :radius)(send node2 :xpos) (send node2 :ypos) 
	(sneps:isbase.n (send node2 :the-node)))	;
    (multiple-value-bind (label-from-x label-from-y)
	;; Establish starting point of the label.
	(calculate-arc-label-position from-x from-y to-x to-y arc-label)
      ;; Draw the arc.
      (send window :draw-arrow
	    (- from-x (first *viewport-position*))
	    (- from-y (second *viewport-position*))
	    (- to-x (first *viewport-position*))
	    (- to-y (second *viewport-position*))
	    alu)
      ;; Label the arc.
      (send window :draw-string
	    (string arc-label)
	    (- label-from-x (first *viewport-position*))
	    (- label-from-y (second *viewport-position*))
	    (- to-x (first *viewport-position*))
	    (- to-y (second *viewport-position*))
	    nil (list ':fix ':roman *font-size*)
	    alu)
      ;; Set up a new mouse-sensitive area on the screen for this arc.
      (send window :primitive-item :new-type self
	    (- (min label-from-x to-x)
	       (left *viewport-position*))
	    (- (min to-y label-from-y)
	       (top *viewport-position*)
	       10)
	    (- (max label-from-x to-x)
	       (left *viewport-position*))
	    (- (max to-y label-from-y)
	       (top *viewport-position*))))))


(defmethod (arc :draw-moving-arc) (window)
  (multiple-value-bind (from-x from-y to-x to-y)
      (calculate-node-edges
	(send node1 :radius) (send node1 :xpos) (send node1 :ypos)
	(sneps:isbase.n (send node1 :the-node))
	(send node2 :radius) (send node2 :xpos) (send node2 :ypos)
	(sneps:isbase.n (send node2 :the-node)))
    (send window :draw-arrow from-x from-y to-x to-y tv:alu-ior 3)))    
   

(defmethod (arc :make-converse)()
  (let ((temp node1))
    (send self :draw-self *window* tv:alu-andca)
    (setf node1 node2)
    (setf node2 temp)
    (setf arc-label (sneps::converse.r arc-label))
    (send self :draw-self)))

    

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                 ;;;
;;;  GRAPH-WINDOW Flavor & Methods  ;;;
;;;                                 ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defflavor graph-window ()
	   (tv:process-mixin
	     tv:basic-mouse-sensitive-items
	     tv:borders-mixin
	     tv:window))



(defmethod (:draw-arrow graph-window)
	   (from-x from-y to-x to-y &optional (alu tv:alu-ior)(thickness 1))
  "draws a line with an arrow-head at the endpoint."
  (graphics:draw-arrow from-x from-y to-x to-y :stream self :alu alu :thickness thickness))

(defmethod (:draw-bent-arrow graph-window)
	   (from-x from-y mid-x mid-y to-x to-y &optional (alu tv:alu-ior)(thickness 1))
  "draws a spline with an arrow-head at the endpoint."
  (draw-bent-arrow from-x from-y mid-x mid-y to-x to-y :stream self :alu alu :thickness thickness))

(defmethod (:draw-filled-triangle graph-window)
	   (x1 y1 x2 y2 x3 y3 &optional (alu tv:alu-ior))
  "draws a filled in triangle."
  (graphics:draw-triangle x1 y1 x2 y2 x3 y3 :stream self :alu alu :filled t))

(defmethod (:draw-unfilled-ellipse graph-window)
	   (center-x center-y x-radius y-radius &optional (alu tv:alu-ior)(thickness 1))
  "Draws an unfilled ellipse."
  (graphics:draw-ellipse center-x center-y x-radius y-radius
			 :stream self :alu alu :filled nil :thickness thickness))

(defmethod (:draw-unfilled-circle graph-window)
	   (center-x center-y radius &optional (alu tv:alu-ior)(thickness 1))
  "Draws an unfilled circle."
  (graphics:draw-circle center-x center-y radius
			:stream self :alu alu :filled nil :thickness thickness))


(defmethod (:display graph-window)()
  (dolist (node *displayed-nodes* t)
    (when (send node :in-view)
      (send node :draw-self)
      (dolist (arc (send node :arcs) t)
	(send arc :draw-self)))))






(tv:add-typeout-item-type *alist-alpha*
			  :new-type "EXIT" exit
			  nil "Exit and Kill Ginseng Window.")

(tv:add-typeout-item-type *alist-alpha*
			  :new-type "RESET" reset
			  nil "Reset Ginseng Window to its initial state.")

(tv:add-typeout-item-type *alist-alpha*
			  :new-type "SET ATTRIBUTES" attrib
			  nil "Window's Display Attributes Menu.")

(tv:add-typeout-item-type *alist-alpha*
			  :new-type "REFRESH" refresh
			  nil "Clear and Redraw Ginseng Window.")

(tv:add-typeout-item-type *alist-alpha*
			  :new-type "TOGGLE PANNING" pan
			  nil "Toggle (ON/OFF) Panning Mode.")

(tv:add-typeout-item-type *alist-alpha*
			  :new-type "DRAW CONVERSE ARC" conv
			  nil "Replace this arc with its converse.")

(tv:add-typeout-item-type *alist-alpha*
			  :new-type "DELETE" delete
			  nil "Delete the graphical representation of this Node.")

(tv:add-typeout-item-type *alist-alpha*
			  :new-type "ERASE" erase
			  nil "Erase the display of this Node -- Temporarily.")

(tv:add-typeout-item-type *alist-alpha*
			  :new-type "DUMP" dump
			  nil "Dump this Node.")

(tv:add-typeout-item-type *alist-alpha*
			  :new-type "DESCRIBE" desc
			  nil "Describe this Node.")

(tv:add-typeout-item-type *alist-alpha*
			  :new-type "MOVE NODE/BEND ARC" move
			  t "Move this Node, or Bend this Arc.")










    
    




