;;; -*- Mode: Lisp; Syntax: Common-Lisp; Package: SNEPS; Base: 10 -*-

;; Copyright (C) 1984--2004 Research Foundation of 
;;                          State University of New York

;; Version: $Id: dd.lisp,v 1.11 2004/08/26 23:25:46 snwiz Exp $

;; This file is part of SNePS.

;; SNePS is free software; you may redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; SNePS is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with SNePS; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA, or to
;; Dr. Stuart C. Shapiro, Department of Computer Science and Engineering,
;; University at Buffalo, The State University of New York, 
;; 201 Bell Hall, Buffalo, NY 14260, USA

(in-package :sneps)


; ==========================================================================
;
; describe 
; --------
;
;      arguments     : snepsul-exp - <snepsul-exp>
;
;      returns       : <node set>
;
;      description   : This calls "sneval" to evaluate "snepsul-exp" to 
;                      get the desired <node set>. 
;                      It prints the description of each <node> in the 
;                      <node set> that has not yet been described during 
;                      the process; the description includes the 
;                      description of all <node>s dominated by the <node>.
;                      It returns the <node set>.
;
;      implementation: Stores the <node>s which have already been described 
;                      in "describe-nodes".
;                      Before tracing the description of a <node>, it 
;                      checks whether the <node> was already been described 
;                      to avoid describing the same <node> repeatedly. 
;                      The variable "describe-nodes" is updated by "des1".
;
;      side-effects  : Prints the <node>'s descriptions.
;
;                                         written:  CCC 07/28/83
;                                         modified: CCC 09/26/83
;                                                   ejm 10/10/83
;                                                   njm 09/28/88
;                                                   njm  4/27/89
;                                                    hc  1/06/92
;                                                    hc  7/18/93
;
(defsnepscom describe ((&rest snepsul-exp) (top ns bns tbns fns))
  (declare (special outunit))
  (let* ((crntct (processcontextdescr snepsul-exp))
	 (ns (in-context.ns (nseval (getsndescr snepsul-exp))
			    crntct))
	 (described-nodes (new.ns))
	 (full nil))
    (declare (special crntct described-nodes full))
    (when ns
      (terpri outunit)
      (mapc #'(lambda (n)
		(if (not (ismemb.ns n described-nodes))
		    (PP-nodetree (des1 n) outunit)))
	    ns))
    (values ns crntct)))


; ==========================================================================
;
; full-describe 
; -------------
;
;      arguments     : snepsul-exp - <snepsul-exp>
;
;      returns       : <node set>
;
;      description   : This calls "sneval" to evaluate "snepsul-exp" to 
;                      get the desired <node set>. 
;                      It prints the description of each <node> in the 
;                      <node set> that has not yet been described during 
;                      the process; the description includes the 
;                      description of all <node>s dominated by the <node>
;                      and the description of the support of the node, if 
;                      it exists.
;                      It returns the <node set>.
;
;      implementation: Stores the <node>s which have already been described 
;                      in "describe-nodes".
;                      Before tracing the description of a <node>, it 
;                      checks whether the <node> was already been described 
;                      to avoid describing the same <node> repeatedly. 
;                      The variable "describe-nodes" is updated by "des1".
;                      The variable "full" informs "des1" that a description 
;                      of the support of nodes is required. 
;
;      side-effects  : Prints the <node>'s descriptions.
;
;                                         written :  njm 09/28/88
;                                         modified:  njm 04/27/89
;                                                     hc  1/06/92
;                                                     hc  7/18/93
;
(defsnepscom full-describe ((&rest snepsul-exp) describe)
  (declare (special outunit))
  (let* ((crntct (processcontextdescr snepsul-exp))
	 (ns (in-context.ns (nseval (getsndescr snepsul-exp))
			    (iscontext-given snepsul-exp)))
	 (described-nodes (new.ns))
	 (full T))
    (declare (special described-nodes crntct full))
    (when ns
      (terpri outunit)
      (mapc #'(lambda (n)
		(if (not (ismemb.ns n described-nodes))
		    (PP-nodetree (des1 n) outunit)))
	    ns))
    (values ns crntct)))
 
;
; ==========================================================================
;
; des1 
; ----
;
;      arguments     : n - <node>
;
;      returns       : <sequence> 
;
;      nonlocal-vars : described-nodes - <node set> - defined in "describe"
;                      full            - <boolean>  - defined in "describe"
;
;      description   : This checks if the <node> has been described or not.  
;                      If not, it puts the <node> in "describe-nodes" and 
;                      gets the <cable set> of the <node>. 
;                      For each <cable> in the <cable set>, if the 
;                      <relation> is an ascending <relation>, it does 
;                      nothing.
;                      If not an ascending <relation> it lists the 
;                      <relation> and the description(s) of the <node set> 
;                      which the <relation> points to.
;                      To get the description(s) of the <node set>, "des1" 
;                      calls itself for each <node> in the <node set>.
;                      Before tracing the description of a <node>, it 
;                      checks if the <node> has already been described 
;                      to avoid describing the same <node> repeatedly.
;                      Depending on the value of the variable "full", a
;                      description of the node's support is can be appended 
;                      to the end of its description. 
;                      
;      side-effects  : Variable "described-nodes" is side-effected.
;
;                                         written:  CCC 07/28/83
;                                         modified: CCC 09/26/83
;                                                   ssc 02/11/87
;                                                   njm 09/28/88
;
;
(defun des1 (n)
  (declare (special described-nodes full))
  (cons (describe.n n)
	(unless (ismemb.ns n described-nodes)
	  (setq described-nodes (insert.ns n described-nodes)) 
	  ;;
	  ;;  describes the cable set
	  ;;
	  (append 
	    (mapcan #'(lambda (c)
			(if (not (isup.r (relation.c c)))
			    (list
			      (cons (relation.c c)
				    (mapcar #'(lambda (n1)
						(cond ((or (isbase.n n1) (isvar.n n1))
						       (setq described-nodes
							     (insert.ns n1 described-nodes))
						       (cond ((isnumber.n n1)
							      (node-to-number.n n1)) (t n1)))
						      (t (des1 n1))))
					    (nodeset.c c))))))
		    (n-to-cs n))
	    ;;
	    ;;  describes the context cable set
	    ;;
	    (if full
		(descrcontext-cable-set (node-asupport n)))))))


;
; ==========================================================================
;
; dump 
; ----
;
;      arguments     : snepsul-exp - <snepsul-exp>
;
;      returns       : <node set>
;
;      description   : It calls "sneval" to evaluate "snepsul-exp" to get 
;                      the desired <node set>.  It prints out the <node> 
;                      together with its <cable set> for each <node> in 
;                      the <node set>.
;
;      side-effects  : Prints out the <node> and its <cable set>.
;                     
;
;                                         written:  CCC 07/28/83
;                                         modified: CCC 09/26/83
;                                                   ejm 10/10/83
;                                                   njm 04/27/89
;                                                    hc  1/06/92
;                                                    hc  7/18/93
;
(defsnepscom dump ((&rest snepsul-exp) describe)
  (declare (special outunit))
  (let* ((crntct (processcontextdescr snepsul-exp))
	 (ns (in-context.ns
	      (nseval (getsndescr snepsul-exp))	       
	      crntct)))
    (declare (special crntct))
    (when ns
      (terpri outunit)
      (mapc #'(lambda (n)
		(princ (list n (node-fcableset n)) outunit)
		(terpri outunit))
	    ns))
    (values ns crntct)))
 
;
; ==========================================================================
;
; PP-nodetree
; -----------
;
;       arguments     : nodetree - <nodetree>
;                       stream - an output stream
;
;       returns       : <never mind>
;
;       description   : Pretty print the given <nodetree> to <stream>.
;
;       side-effects  : Pretty print the <nodetree>.
;
;                                          written : CCC 09/26/83
;                                          modified: hc 10/2/90
;                                                        1/6/92
;                                                    aec 1/23/96
 
(defun PP-nodetree (nodetree &optional (stream outunit))
  (format stream "~&")
  (write nodetree :stream stream :pretty t :escape nil)
  (terpri stream))
 
(defsnepscom surface ((&rest snepsul-exp))
  "Hands its argument node set to the generation grammar starting in state G."
  (declare (special outunit parser:*atn-arcs-hashtable* englex:*lexicon*))
  (if (not (initialized-p englex:*lexicon*))
      (warn "No lexicon is loaded. You should load a lexicon via~
           ~%`(lexin \"<lexicon-filename>\")' before using `surface'."))
  (if (not (initialized-p parser:*atn-arcs-hashtable*))
      (sneps-error
       (format
	nil "No ATN grammar is loaded. You must load a grammar via~
           ~%`(atnin \"<atn-grammar-filename>\")' before using `surface'.")
       'describe 'surface))
  (let* ((parser:*trace-level* -1)
	 (parser:*all-parses* nil)
	 (response
	  (parser:flatten
	   (parser::internal-parse (nseval snepsul-exp) 'snepsul::G))))
    (declare (special parser:*trace-level* parser:*all-parses*))
    (when response
      (format outunit
	      "~%~a.~%"
	      (string-trim "()" (princ-to-string response))))
    (values)))


(defun  slight-surface (node &optional (stream nil))
  (format stream "~A" node))

(defun format.ns (nsf)
  "Formats a nodesetform that contains either infix or postfix operations into 
   prefix form for evaluation."
  (declare (special nsf))
  (loop
    (cond ((eq (second nsf) '!)
	   (setf nsf
		 (cons (list '! (first nsf)) (cddr nsf))))
	  ((isrearrange.com (second nsf))
	   (cond ((third nsf)
		  (setf nsf
			(cons (list (second nsf) (first nsf) (third nsf)) (cdddr nsf))))
		 (t (sneps-error (format nil
					 "Infix operator missing second operand: ~A"
					 (cdr nsf))
				 '|Evaluation of node description|
				 'format.ns))))
	  (t (return nsf)))))

(defun ddeval (snepsul-exp)
    (let (result)
      (declare (special result))
      (setq result
	    (cond ((null snepsul-exp) (new.ns))
		  ((numberp snepsul-exp) (ddeval (un-ize snepsul-exp)))
		  ((atom snepsul-exp) (nseval snepsul-exp))
		  ((isfns.com (first snepsul-exp))
		   ((lambda (ev-result)
		      (if (atom ev-result) (ddeval ev-result) ev-result))
		    (eval (find-inside snepsul-exp))))
		  (t (nseval snepsul-exp))))
      result))

; ==========================================================================
;
; descrcontext-cable-set 
; ----------------------
;
;      arguments     : ctcs - <context cable set>
;
;      returns       : <sequence>
;
;      description   : it lists the description of the <context cable set> 
;                      "ctcs".
;
;
;                                         written :  njm 09/28/88
;                                         modified: 
;                                                   
;
(defun descrcontext-cable-set (ctcs)
  "Lists the description of the <context cable set> `ctcs'"
  (cond ((isnew.ctcs ctcs) nil)
	(t (append (descrcontext-cable (ot.ctcs ctcs)
						 (contextset.ctcs ctcs))
		   (descrcontext-cable-set (others.ctcs ctcs))))))



; ==========================================================================
;
; descrcontext-cable 
; ------------------
;
;      arguments     : otag - <otag>
;                      cts - <context set>
;
;      returns       : <sequence>
;
;      description   : it lists the description of the <context cable> 
;                      defined by the "otag" and "cts".
;
;
;                                         written :  njm 09/28/88
;                                         modified: 
;                                                   
;
(defun descrcontext-cable (otag cts)
  "Lists the description of the <context cable> defined by `otag'
   and `cts'"
  (cond ((isnew.cts cts) nil)
	(t (cons (list otag (descrcontext (choose.cts cts)))
		   (descrcontext-cable otag (others.cts cts))))))



;;;========================= Code for show ===============================


    ;;; ==========================================================================
;;;
;;; parse-show-params
;;; -----------------
;;;
;;;      arguments     : snepsul-exp - <snepsul-exp>
;;;
;;;      returns       : list of the form:
;;;                      (list-of-nodes file-name
;;;                          output-format is-file-name-generated)
;;;
;;;      description   : this function attempts to locate the optional
;;;                      params to "show" command (i.e. :file and :format)
;;;                      if one or both of the parameters are not present
;;;                      their values are set to the default ones
;;;
;;;      implementation: if :file parameter is not found, a temp file
;;;                      name in /tmp directory is generated
;;;                      if :format parameter is not found, it is
;;;                      set to "gif"
;;;                      is-file-name-generated is needed to inform
;;;                      "show" that a file in /tmp will be created
;;;                      and will have to be deleted
;;;
;;;      side-effects  : none
;;;
;;;                                         written: ddligach 03/11/2004
;;;
(defun parse-show-params (snepsul-exp)
  (let ( ;; get optional file name param
	(file-param (member :file snepsul-exp))
	;; get optional format param
	(format-param (member :format snepsul-exp))) 
    (list 
     ;; extract nodes
     (butlast snepsul-exp 
              (max (length file-param) (length format-param)))

     ;; extract file parameter value
     (if (null (second file-param))
	 (string (gensym "/tmp/sneps-graph-")) ; default file name
       (string (second file-param)))

     ;; extract format parameter value
     (if (null (second format-param))
	 (string 'gif)			; default format
       (string (second format-param)))

     ;; indicate whether file name was passed as a param 
     ;; (and therefore, whether a temporary file 
     ;; in /tmp directory was generated)
     (null (second file-param)))))

;;; ==========================================================================
;;;
;;; show
;;; ----
;;;
;;;      arguments     : snepsul-exp - <snepsul-exp>
;;;
;;;      returns       : <node set>
;;;
;;;      description   : This command is similar to describe. However, instead
;;;	                   of printing the description of each node, it produces
;;;                      a graphical representation of the network.
;;;                      This command takes two optional parameters:
;;;             
;;;                      :file <file_name> 
;;;                      :format <ps or gif>
;;;
;;;                      If no file name is specified, a temporary file
;;;                      is generated in /tmp directory.
;;;                      Default output format is "gif"
;;;                      Sample calls:
;;;                      (show m1 m2 m3)
;;;                      (show *nodes)
;;;                      (show m7 m8 :format ps)
;;;                      (show *nodes :file tmpFileName :format ps)
;;;
;;;      implementation: The implementation is similar to describe. 
;;;		         The difference is that instead of printing
;;;                      the description of each node, it opens
;;;                      a file and calls "generate-dot-file", which writes 
;;;                      a representation of the network in the DOT language. 
;;;                      Subsequently, the 'dot' compiler is invoked and an
;;;                      output file (gif or ps) is produced. The output file
;;;                      is displayed using xv or gv.
;;;
;;;      side-effects  : none
;;;
;;;                                         written: ddligach 02/21/2004 
;;;
(defsnepscom show ((&rest snepsul-exp) (top ns bns tbns fns))
  (declare (special outunit))
  ;; get gif and dot file mames
  (let* ((param-list (parse-show-params snepsul-exp))
         (exp (first param-list))	; set of nodes
         (file-name (second param-list)) ; file name
         (output-format (third param-list)) ; output file format
         (temp-file-generated (fourth param-list)) ; tmp file present?
	 (dot-file-name (merge-pathnames file-name "*.dot")) ; dot file
	 (output-file-name
	  (merge-pathnames file-name 
			   (format nil "*.~A" output-format)))) ; out file
    ;; open file and write network specs in "dot" language
    (with-open-file (dot-stream dot-file-name
		     :direction :output
		     :if-exists :supersede
		     :if-does-not-exist :create)
      (format dot-stream "digraph RE { ~%") ; write opening line to file
      ;; get nodes and write them to file (similar to "describe")
      ;; Original limited nodes shown to those in the context.
      ;;   I suspect this was just following what describe does.
      ;;   I changed that to allow all requested nodes to be shown.
      (let* ((crntct (processcontextdescr exp))
	     ;; (ns (in-context.ns (nseval (getsndescr exp)) crntct))
	     (ns (nseval (getsndescr exp)))
	     (described-nodes (new.ns))
	     (full nil))
	(declare (special crntct described-nodes full))
	(when ns
	  (terpri outunit)
	  (mapc #'(lambda (n)
		    (if (not (ismemb.ns n described-nodes))
			(generate-dot-file (des1 n) dot-stream)))
		ns))
	(values ns crntct))
      ;; done writing to file, now display the results
      (format dot-stream "} ~%"))	; write closing line to file
    ;; generate output file (gif or ps)
    ;; display it (with xv or gv)
    ;; remove it if it is a temporary file
    (excl:shell
     (format nil "dot -T~A ~A > ~A; ~A ~A; ~A &" 
	     output-format dot-file-name output-file-name
	     (if (equal output-format "ps") 
		 "gv" 
	       "xv") output-file-name
	     (if temp-file-generated
		 (format nil "/bin/rm -f ~A.*" file-name)
	       ""))))) 

;;; ==========================================================================
;;;
;;; generate-dot-file
;;; -----------------
;;;
;;;      arguments     : nodetree - <nodetree>
;;;                      dot-stream - an output stream
;;;
;;;      returns       : m-node
;;;
;;;      description   : Converts <nodetree> to dot language and 
;;;                      writes it to <stream> 
;;;
;;;      implementation: This function assumes that <nodetree> has
;;;                      the following general structure:
;;;                      (m-node (arc node* nodetree* )* )
;;;                         where '*' is a kleene star
;;;                      The outter loop walks through every element of 
;;;                      the nodetree. The inner loop parses each particular
;;;                      element of the nodetree and writes its 'dot'
;;;                      representation to dot-stream.
;;;                      General 'dot' file format:
;;;                          A -> B [label="C"]
;;;                          A and B are nodes, C is an ark between A and B
;;;
;;;      side-effects  : Writes to file the network's representation in
;;;                      'dot' language.
;;;
;;;                                         written: ddligach 02/21/2004
;;;
(defun generate-dot-file (nodetree dot-stream)
  (let ((m-node (car nodetree))) 
    (when (null (cdr nodetree)) ; handle the case of showing a base node
      (format dot-stream "\"~A\"; ~%" m-node))
    ;; loop through nodetree elements i.e. (arc node ...)
    (loop for list-elem in (cdr nodetree) 
			   ;; loop through each node in (arc node1 node2 ...)
	do (loop for node in (cdr list-elem)
	       do (format dot-stream "\"~A\" -> \"~A\" [label=\"~A\"]; ~%"
			  m-node	; match first '~A'
			  (if (atom node) 
			      node	; match second '~A' 
					; this is not an atomic node
					; so, it must be a nodetree
					; make a recursive call
			    (generate-dot-file node dot-stream))
			  (car list-elem)))) ; match third '~A' (an ark)
    ;; return m-node
    (values m-node)))

    




