;;; -*- Mode: Lisp; Syntax: Common-Lisp; Package: SNEPS; Base: 10 -*-

;; Copyright (C) 1984--2004 Research Foundation of 
;;                          State University of New York

;; Version: $Id: with-snepsul.lisp,v 1.13 2004/08/26 23:25:47 snwiz Exp $

;; This file is part of SNePS.

;; SNePS is free software; you may redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; SNePS is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with SNePS; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA, or to
;; Dr. Stuart C. Shapiro, Department of Computer Science and Engineering,
;; University at Buffalo, The State University of New York, 
;; 201 Bell Hall, Buffalo, NY 14260, USA

(in-package :sneps)


;; #! with-snepsul reader macro for easy use of SNePSUL expressions
;; inside Lisp functions:
;;
;; Syntax: #!n(snepsul-form1 .... snepsul-formN)
;;
;; Created: hc, Feb-4-92
;;
;; Reader macro that does away with the never ending blues of people
;; trying to execute SNePSUL commands within lisp functions. The state
;; of the art so far was something like this:
;;
;;   (defun myassert (relation nodes)
;;     (eval `(snepsul::assert ,(intern (string relation) 'snepsul)
;;                             ,@nodes)))
;;
;; with the new reader macro this function can be defined as
;;
;;   (defun myassert (relation nodes)
;;     #!((assert ~relation ~@nodes)))
;;
;; I.e., the form following #! is taken to be a list of SNePSUL fomrs,
;; each of which will be executed just as if it had been typed that
;; way at the SNePS prompt, regardless of the package in which
;; myassert is defined. References to Lisp variables can be made via a
;; ~ reader macro mechanism (similar to the backquote syntax).
;; Results of ~ expansions will be automatically interned into the
;; SNePSUL package (i.e., any symbols that might be part of such a
;; result), unless explicitly specified otherwise. All the special
;; reader syntax available at the SNePS top-level is available, too.
;;
;; Here is the semantics of the ~ syntax:
;; ~<s-expression> :  S-expression will be read with ordinary reader syntax
;;                    and at execution time it will be evaluated and its
;;                    value inserted into the SNePSUL expression. If the
;;                    value is a symbol or a list containing symbols then these
;;                    symbols will be interned into the SNePSUL package first.
;;                    Ex: #!((describe ~'(m1 m2))) will create
;;                           (describe (m1 m2)) as its result.
;; ~@<s-expression> : Just like ~ but the value of the s-expression has to
;;                    be a list which will be spliced into the SNePSUL
;;                    expression. Any symbols occuring as leaves in the list
;;                    will be interned into the SNePSUL package first.
;;                    Ex: #!((describe ~@'(m1 m2))) will create
;;                           (describe m1 m2) as its result.
;; ~~<s-expression> : Just like ~ but symbols in the value will not be interned
;;                    into the SNePSUL package:
;; ~~@<s-expression>: Just like ~@ but symbols in the value will not be
;;                    interned into the SNePSUL package:
;;
;; CAUTION: The ~ syntax can only be used within SNePSUL forms, but
;; not to denote multiple forms, e.g., while #!(~com1 ~com2 ~com3) is
;; legal (as long as the runtime values of comX represent proper
;; snepsul commands), #!(~@commands) is not!!
;;
;; Supplying an optional digit argument can be used to select a
;; specific evaluation function, or to suppress output:
;;
;;   arg       eval function      silent        syntax
;; no arg       topsneval           no         #!(....)
;;   1          topsneval           no         #1!(....)
;;   2          eval                no         #2!(....)
;;   3          topsneval           yes        #3!(....)
;;   4          eval                yes        #4!(....)
;;
;; For example, #4!((build relation node)) will use the function eval
;; to evaluate the form (hence build can be used!!), and will suppress
;; any output generated by the snepsul command.
;;
;; Here is an example that shows how #! generates SNePSUL expressions:
#|
> (in-package 'user)
#<Package "USER" 79D15E>

> (defun myassert (relation nodes)
    (let ((base-node-var 'mybase))
      #!((define ~relation ~~relation myrel snip::test)
	 (assert ~relation (~@nodes) snip::test #~base-node-var)
	 (describe ~@(setq nodes (cdr nodes)))
	 (assert ~~relation (~~@nodes) myrel *~base-node-var)
	 (assert arg (build myrel hans ~relation franz)
		 myrel *mybase)
	 (describe *nodes))))
MYASSERT

;; If the variable *with-snepsul-eval-function* is bound to the
;; function #'with-snepsul-trace-eval then the generated SNePSUL
;; expression will only be printed, but not actually executed:

> (let ((sneps:*with-snepsul-eval-function*
          #'sneps:with-snepsul-trace-eval))
    (myassert 'relrel '(hans franz otto)))
(SNEPS:DEFINE SNEPSUL::RELREL ;; Note "snepsulization" with single ~
              USER::RELREL    ;; Package preservation with double ~
              SNEPSUL::MYREL  ;; Unqualified symbols go into SNePSUL
              SNIP::TEST)     ;; Qualified symbols keep their package
(SNEPS:ASSERT SNEPSUL::RELREL
              (SNEPSUL::HANS SNEPSUL::FRANZ
                             SNEPSUL::OTTO)
              SNIP::TEST
	      ;; had to replace the |'s with !'s here (comment problem)
              (SNEPS:!#! 'SNEPSUL::MYBASE))    ;; Combination of # and ~
(SNEPS::DESCRIBE SNEPSUL::FRANZ SNEPSUL::OTTO)
(SNEPS:ASSERT USER::RELREL (USER::FRANZ USER::OTTO)
              SNEPSUL::MYREL (SNEPS:* 'SNEPSUL::MYBASE))
(SNEPS:ASSERT SNEPS:ARG (SNEPS:BUILD SNEPSUL::MYREL SNEPSUL::HANS
				     SNEPSUL::RELREL SNEPSUL::FRANZ)
              SNEPSUL::MYREL (SNEPS:* 'SNEPSUL::MYBASE))
(SNEPS::DESCRIBE (SNEPS:* 'SNEPS:NODES))

;; Now actually run it:
> (myassert 'relrel '(hans franz otto))
(FRANZ)
(OTTO)

(B1)
(FRANZ)    ;; user::franz
(FRANZ)    ;; snepsul::franz
(HANS)
(M1! (RELREL FRANZ HANS OTTO))
(M2! (RELREL FRANZ OTTO))
(M3 (MYREL HANS) (RELREL FRANZ))
(M4! (ARG (M3)))
(M5! (RELREL FRANZ HANS OTTO)
     (TEST B1))
(M6! (MYREL B1)
     (RELREL FRANZ OTTO))
(M7! (ARG (M3)) (MYREL B1))
(B1 FRANZ FRANZ HANS M1! M2! M3 M4! M5! M6! M7! OTTO OTTO)
SNEPS:DEFAULT-DEFAULTCT

;; Here's what the definition of myassert looks like:
> (ppdef 'myassert)
(LAMBDA (RELATION NODES)
  (BLOCK MYASSERT
    (LET ((BASE-NODE-VAR 'MYBASE))
      (PROGN ;; progn generated by #!
	(SNEPS::WITH-SNEPSUL-EVAL
	    `(SNEPS:DEFINE ,(SNEPS::SNEPSULIZE RELATION) ,RELATION
	      SNEPSUL::MYREL SNIP::TEST)
	  #'SNEPS:TOPSNEVAL NIL)
	(SNEPS::WITH-SNEPSUL-EVAL
	    `(SNEPS:ASSERT ,(SNEPS::SNEPSULIZE RELATION)
	      (,@(SNEPS::SNEPSULIZE NODES)) SNIP::TEST
	      (SNEPS:!#! ',(SNEPS::SNEPSULIZE BASE-NODE-VAR)))
	  #'SNEPS:TOPSNEVAL NIL)
	(SNEPS::WITH-SNEPSUL-EVAL
	    `(SNEPS::DESCRIBE
	      ,@(SNEPS::SNEPSULIZE (SETQ NODES (CDR NODES))))
	  #'SNEPS:TOPSNEVAL NIL)
	(SNEPS::WITH-SNEPSUL-EVAL
	    `(SNEPS:ASSERT ,RELATION (,@NODES) SNEPSUL::MYREL
	      (SNEPS:* ',(SNEPS::SNEPSULIZE BASE-NODE-VAR)))
	  #'SNEPS:TOPSNEVAL NIL)
	(SNEPS::WITH-SNEPSUL-EVAL
	    `(SNEPS:ASSERT SNEPS:ARG
	      (SNEPS:BUILD SNEPSUL::MYREL SNEPSUL::HANS
	       ,(SNEPS::SNEPSULIZE RELATION) SNEPSUL::FRANZ)
	      SNEPSUL::MYREL (SNEPS:* 'SNEPSUL::MYBASE))
	  #'SNEPS:TOPSNEVAL NIL)
	(SNEPS::WITH-SNEPSUL-EVAL
	    '(SNEPS::DESCRIBE (SNEPS:* 'SNEPS:NODES))
	  #'SNEPS:TOPSNEVAL NIL)))))
>
|#

;; Implementation:

;; Create a completely emtpy package. If this package is the current package
;; while some s-expression gets printed, then every symbol that is part of
;; that expression will be prefixed with its home package (otherwise, symbols
;; that are members of the current package would be printed without
;; package information)
(unless (find-package 'empty)
  (make-package 'empty :use nil))

;; This structure is a hack to generate special printed
;; representations for s-expressions. How a certain s-expression will
;; be printed is controlled by the type slot, for example, objects
;; read with the ~ macro will be printed as ,object.  Printing these
;; to a string (with a backquote) and reading them again expands them
;; into a structure similar to the one obtained by normal backquote
;; syntax, but without having to use an actual backquote.  Without
;; this we would need to construct "by hand" the same deeply nested
;; list/append structure that is constructed by the normal backquote
;; macro, which is kind of messy.
(defstruct (special-object
	     (:print-function
	       (lambda (object stream depth)
		 (declare (ignore depth))
		 (let ((sexp (special-object-s-expression object))
		       (type (special-object-type object)))
		 ;; Using the empty package here will prefix every
		 ;; symbol with an appropriate package
		 (let ((*package* (find-package 'empty)))
		   (case type
		     (:normal (format stream ",~s" sexp))
		     (:splice (format stream ",@~s" sexp))))
		 ))))
  (s-expression) ;; the s-expression that should be printed specially
  (type))        ;; either :splice or :normal

(defun snepsulize (s-expression)
  "Takes an arbitrary S-EXPRESSION and returns an identical copy with
all symbols interned into the SNEPSUL package. Changes only take place if
S-EXPRESSION was a symbol or a list (a tree)."
  (substitute-tree
   #'symbolp
   #'(lambda (leaf)
       (intern (symbol-name leaf) 'snepsul))
   s-expression))

(defvar *tilde-sneps-readtable* (copy-readtable *sneps-readtable*)
  "A SNePS readtable plus the special ~ reader macro")

;; The ~ macro is necessary to be able to use the normal readtable to
;; read the following Lisp s-expression. Using a standard backquoted
;; expression and relying on the standard implementation of , and ,@
;; would not work, because then everything would be read with the
;; sneps readtable when we read the body of #!.
(let ((*readtable* *tilde-sneps-readtable*))
  ;; This reader macro reads the next s-expression and creates a
  ;; special-object structure with :s-expression set to the object read,
  ;; and :type set to :splice or :normal depending on whether a @ is
  ;; the next character or not. S-expressions will be embedded into a
  ;; snepsulize call unless a ~~ syntax was used.
  (set-macro-character #\~ #'(lambda (stream char)
			       (declare (ignore char))
			       ;; Objects within ~ syntax are read with
			       ;; standard readtable.
			       (let ((*readtable* *original-readtable*)
				     snepsulize type s-expression)
				 ;; If the next non-whitespace character is
				 ;; ~ then don't snepsulize
				 (setq snepsulize
				       (case (peek-char t stream t nil t)
					 (#\~ (read-char stream t nil t) nil)
					 (t   t)))
				 ;; If the next non-whitespace character is
				 ;; @ then splice the result
				 (setq type 
				       (case (peek-char t stream t nil t)
					 (#\@ (read-char stream t nil t)
					      :splice)
					 (t   :normal)))
				 ;; Now read the actual object
				 (setq s-expression (read stream t nil t))
				 ;; Create a special-object structure
				 (make-special-object
				  :s-expression
				  (cond (snepsulize
					 `(snepsulize ,s-expression))
					(t s-expression))
				  :type type))))
  )

;; It is important to compute the translation of a SNePSUL body at
;; read time, because then the packages in which the expression is
;; read and in which it is expanded are the same. This makes sure that
;; unqualified symbols will print unqualified, and qualified symbols
;; will print with a package prefix.  In the old with-snepsul version
;; we would have problems if the value of *package* at reading time
;; was different from *package* at expansion time. Because we had to
;; print the body to a string first before it could be read with the
;; sneps read table, unqualified symbols could wind up in a different
;; package.  For compiled code this was no problem, because the
;; compiler expanded everything properly, but for interpreted code
;; this strategy did not always work. Now, even for interpreted code
;; the expansion is done at read time which is not only cleaner, but
;; also more efficient (done only once instead of at every invocation
;; of with-snepsul).  Also, reading a SNePSUL body with the standard
;; read table would create all kinds of pkg::~@somevar symbols which
;; were a pain to deal with. Now, using the special tilde read
;; table immediately, makes sure that no such symbols get created.
;; 
(defun with-snepsul-body-reader (stream subchar arg)
  "The macro function for the #! reader macro. It reads the next
s-expression using the special tilde/sneps readtable and assumes the
resulting object is a list of SNePSUL expressions. Every symbol that
is not within the scope of a ~ will be interned into the SNePSUL
package. Each SNePSUL expression will be translated into a backquote
expanded expression which will be passed to an evaluation function
depending on the value of ARG (none, 1 and 3 will use TOPSNEVAL, 2 and
4 will use EVAL). If ARG was 3 or 4 all output generated by the
execution of a SNePSUL expression will be suppressed."
  (declare (ignore subchar))
  (let* ((tilde-sneps-expanded-body
	  (let ((*readtable* *tilde-sneps-readtable*))
	    (read stream t nil t)))
	 ;; Print the expanded body as a string. Where we had ~ before
	 ;; we now have special-object structures which print as
	 ;; ,s-exp or ,@s-exp if we had a splice macro. Every form
	 ;; will be printed inside a (with-snepsul-eval `<form>
	 ;; <evalfn> <silent>) structure which will generate the
	 ;; proper backquote expanded SNePSUL forms once we read the
	 ;; string.
	 (tilde-sneps-expanded-body-as-string
	  (format nil (format nil
			      "(~~{(~a `~~s #'~a ~s)~~})"
			      "sneps::with-snepsul-eval"
			      (case arg
				((nil 1 3) "sneps:topsneval")
				((2 4) "lisp:eval"))
			      (case arg
				((nil 1 2) nil)
				((3 4) t)))
		  tilde-sneps-expanded-body))
	 ;; Finally, read the expanded string into the SNePSUL
	 ;; package. All unqualified symbols will be interned into
	 ;; snepsul. All qualified symbols that are not members of the
	 ;; current package will keep their package. All symbols
	 ;; inside ~ constructs were printed with package prefixes,
	 ;; hence they will refer to the proper variables.
	 (tilde-sneps-package-expanded-body
	  (let ((*package* (find-package 'snepsul)))
	    (read-from-string tilde-sneps-expanded-body-as-string)))
	 )
    ;; Generate an executable progn:
    (cons 'progn tilde-sneps-package-expanded-body)))

;; Modify the standard read table and the SNePS copy of it:
(set-dispatch-macro-character #\# #\! #'with-snepsul-body-reader
			      *readtable*)
(set-dispatch-macro-character #\# #\! #'with-snepsul-body-reader
			      *original-readtable*)

(defun maybe-call-and-setup-sneps ()
  "SNePS must be initialized properly before WITH-SNEPSUL-EVAL can be called.
This function is a hack that actually calls SNePS and immediately exits it
if SNePS has not yet been initialized. Doing it this way avoids having to
find and bind the various special variables used in the function SNEPS-SETUP."
  (when *sneps-setup-flag*
    (let ((*package* (find-package 'snepsul)))
      ;; After calling SNePS once *sneps-setup-flag* will be NIL
      (with-input-from-string (s "(lisp)")
	(chew-up-output
	 ;; Some lisps might require to actually suppress *terminal-io*, too.
	 (outunit *standard-output*)
	 (let ((*standard-input* s))
	   (sneps)))))))

(defun with-snepsul-standard-eval (function form)
  "Standard function used by with-snepsul-eval to evaluate FORM with
evaluation FUNCTION."
  (funcall function form))

(defun with-snepsul-trace-eval (function form)
  "Does not actually evaluate FORM, only prints it for debugging purposes."
  (declare (ignore function))
  (let ((*package* (find-package 'empty)))
    (pprint form)))

(defun with-snepsul-toplevel-style-eval (function form)
  "Evaluates the SNePSUL FORM using FUNCTION and returns the result.
Additionally, prints prompt, FORM, result and timing information just
like the top-level SNePS loop does."
  (let (;; Bind these so we will also see
	;; execution of SILENT stuff
	(*standard-output* *terminal-io*)
	(outunit *terminal-io*)
	oldtime newtime)
    (format outunit "~a" *sneps-prompt*)  ;;; *prompt* -> *sneps-prompt*  FLJ 9/2/02
    (write form :stream outunit :pretty t :case :downcase)
    (terpri outunit)
    (protect-eval
     (setq oldtime (list (get-internal-run-time) 0))
     (multiple-value-bind (result result-context)
	 (funcall function form)
       (setq newtime (list (get-internal-run-time) 0))
       (let ((crntct result-context))
	 (declare (special crntct))
	 (format outunit "~&~%")
	 (sneps-print result)
	 (set.sv 'lastcommand form)
	 (sneps-timer oldtime newtime)
	 (values result result-context)
	 )))))

(defvar *with-snepsul-eval-function* #'with-snepsul-standard-eval
  "The value of this variable has to be a function of two arguments,
an EVAL-FUNCTION and a FORM to which the function should be applied.
Allowing to bind this variable to different functions allows for various
different behaviors, such as normal evaluation, tracing, top-level like
echoing, evaluating and printing the result, etc., when the FORM gets
evaluated inside WITH-SNEPSUL-EVAL.")

(defun with-snepsul-eval (form &optional (eval-function #'topsneval) silent)
  "Evaluates a FORM (a SNePSUL command) with EVAL-FUNCTION in a proper SNePS
environment. If SILENT is nonNIL all output will be suppressed. Does proper
SNePS setup if necessary. The value of *WITH-SNEPSUL-EVAL-FUNCTION* is then
called to evaluate the FORM and its result will be returned."
  (maybe-call-and-setup-sneps)
  (let ((*package* (find-package 'snepsul))
	(real-outunit (get-default-stream outunit *standard-output*)))
    (chew-up-output
     (outunit)
     (let ((outunit (cond (silent outunit)
			  (t real-outunit))))
       (funcall *with-snepsul-eval-function* eval-function form)))))



    
    




