;;; -*- Mode: Lisp; Syntax: Common-Lisp; Package: SNEPSLOG; Base: 10 -*-

;; Copyright (C) 1984--2004
;; Research Foundation of State University of New York

;; Version: $Id: commands.lisp,v 1.15 2004/08/26 23:25:51 snwiz Exp $

;; This file is part of SNePS.

;; SNePS is free software; you may redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; SNePS is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with SNePS; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA, or to
;; Dr. Stuart C. Shapiro, Department of Computer Science and Engineering,
;; University at Buffalo, The State University of New York, 
;; 201 Bell Hall, Buffalo, NY 14260, USA

(in-package :snepslog)


;; Changes (hc Mar-12-90)
;;  - New function get-def.command that eliminates package problems
;;    with new commands by using the symbol name and interning the
;;    string into the SNEPSLOG package
;;  - Add 'freshline' to execute-snepslog.command to get consistent
;;    newline after output, and add special variable command.string
;;    so ^ and % can use the right readtables.
;;  - change ^ and % to use the right readtables (this should be done
;;    more elegantely, I just hacked the reader code)
;;  - Use with-open-file in demo so aborts won't leave files open.


;
; Implementing snepslog.command 
;


(defmacro def-snepslog.command (command vars &body forms)
  `(setf (get ',command 'snepslog.command)
	 ;; Don't need compile, will be compiled during
	 ;; compilation of command definitions
	 #'(lambda ,vars ,@forms)))

(defun get-def.command (command)
   (and (symbolp command)
	(get (intern (symbol-name command) 'snepslog)
	     'snepslog.command)))

(defun snepslog.command? (form)
  (get-def.command (first form)))

(defun execute-snepslog.command (command &optional command.string)
  (declare (special command.string))
  (apply (get-def.command (first command)) (rest command))
  (format outunit "~&~%")
  (throw 'sneps-error nil))


;
; some useful functions
;

(defun read-eval-print-loop ()
  (format outunit
	  "~&Entering a read/eval/print loop, ~
             type ^^ or end to exit~%")
  (sneps:pseudolisp "--> " '("end" "^^")))

(defun hyp.set? (hyp-set)
  (and (listp hyp-set)
       (every #'wff? hyp-set)
       (issubset.ns (mapcar #'node (wffs->ms hyp-set)) (value.sv 'assertions))))

(defun not.wffs (wff.set)
  (if (listp wff.set)
      (remove-if #'wff? wff.set)
      wff.set))


(defun not.hyps (wff.set &aux (assertions (value.sv 'assertions)))
    (remove-if #'(lambda (wff)
		   (member (node (wff->m wff)) assertions))
	       wff.set))

(defun string.concat (&rest lists)
  "lists is a list os lists. Each of these lists should have a boolean value and a string as elements. If the boolean value is non nil, the string is selected for concatenation. Otherwise, it will not be a part of the final string. For instance:
(string.concat '((t \"this string will be concatenated\") (nil \"This string will not be concatenated\") ('any.value \"this will also be concatenated\"))) would return the string \"this string will be concatenatedthis will also be concatenated\""
  (let ((result ()))
     (dolist (list lists (format nil "~{~a~}" (nreverse result)))
       (when (first list)
	 (setq result (cons (second list) result))))))



(defun check.context.syntax (command &key (name ())
			     (wffs ())
			     &aux (name.ok (and name (symbolp name)))
			     (list.wffs?  (listp wffs))
			     (not.wffs (and list.wffs?
					    (not.wffs wffs)))
			     (not.hyps (and list.wffs?
					    (not.hyps (set-difference wffs not.wffs)))))
  "checks the syntax of the context description. Checks if the name is appropriate and if the wff set is well written"
  (when (or (not name.ok)
	    (not list.wffs?)
	    not.wffs
	    not.hyps)
    (sneps-error (string.concat (list (not name.ok)
				      (format nil
					      "~%The name ~A is not a valid name (a symbol other than nil) for a context."
					      name))
				(and list.wffs?
				     (list not.wffs
					   (format nil
						   "~%The following objects are not wffs: ~A"
						   not.wffs)))
				(and list.wffs?
				     (list not.hyps
					   (format nil
						   "~%The following wffs are not hyps: ~A~%"
						   not.hyps)))
				(list (not list.wffs?)
				      (format nil
					      "~%The following form is not a valid set of wffs: ~A"
					      wffs)))
		 'snepslog
		 command)
    (return-from check.context.syntax t)))

;
; definition of snepslog commands
;

(def-snepslog.command expert ()
  (setq complete-description t))


(def-snepslog.command normal ()
  (setq complete-description nil))


(def-snepslog.command trace ()
  (trace topsneval))


(def-snepslog.command untrace ()
  (untrace topsneval))


(def-snepslog.command trace-atn ()
  (setq trace? t))


(def-snepslog.command untrace-atn ()
  (setq trace? nil))


(def-snepslog.command lisp ()
  (throw :snepslog-end nil))


(def-snepslog.command copyright ()
  (terpri outunit) 
  (sneps:copyright))


(def-snepslog.command % (&rest l)
  (declare (special command.string)
	   (ignore l))
  (let* ((*readtable* sneps::*sneps-readtable*)
	 (sneps-command (read-from-string command.string nil nil :start 1))
	 (*readtable* sneps::*original-readtable*))
    (sneps::sneps-print 
     (topsneval sneps-command))))


(def-snepslog.command ^ (&rest l)
  (declare (special command.string))
  (let ((*readtable* sneps::*original-readtable*))
    (cond (l (prin1
	      (eval (read-from-string command.string nil nil :start 1))
	      outunit))
	  (t (read-eval-print-loop)))))

(def-snepslog.command ^^ ()
  (read-eval-print-loop))


(def-snepslog.command demo (&optional (file 'x file-supplied-p)
				      (pause nil pause-supplied-p))
  (let ((menu (when (sneps:initialized-p
		     sneps:*available-snepslog-demos-menu*)
		sneps:*available-snepslog-demos-menu*)))
    (cond (file-supplied-p
	   (if pause-supplied-p
	       (sneps:demo-internal file :pause pause :menu menu)
	       (sneps:demo-internal file :menu menu)))
	  (t (sneps:demo-internal file :pause 'av :menu menu)))))

(def-snepslog.command set-default-context (name)
  (eval `(set-default-context ,name)))

(def-snepslog.command set-context (context-name hypotesis-set)
  (unless (check.context.syntax 'set-context :name context-name :wffs hypotesis-set)
    (topsneval `(set-context ,(wffs->ms hypotesis-set) ,context-name))))


(def-snepslog.command describe-context (&optional (context-name (value.sv 'defaultct)))
  (unless (check.context.syntax 'describe-context :name context-name)
    (topsneval `(describe-context ,context-name))))


(def-snepslog.command list-context-names ()
  (topsneval `(list-context-names)))


(def-snepslog.command add-to-context (name hyp-set)
  (unless (check.context.syntax 'add-to-context :name name :wffs hyp-set)
    (topsneval `(add-to-context ,(wffs->ms hyp-set) ,name))))

(def-snepslog.command remove-from-context (name hyp-set)
  (unless (check.context.syntax 'remove-from-context :name name :wffs hyp-set)
    #!((remove-from-context ~(wffs->ms hyp-set) ~name))))

(def-snepslog.command clearkb()
  (with-output-to-string (outunit)
    (topsneval '(resetnet)))
  (format outunit "Knowledge Base Cleared"))

(def-snepslog.command clear-infer()
  (format outunit (sneps:clear-infer)))

(def-snepslog.command erase (node)
  (topsneval `(erase ,(wff->m node))))

(def-snepslog.command set-mode-1 ()
  (resetnet t)
  (make.snepslog1))

(def-snepslog.command set-mode-2 ()
  (resetnet t)
  (make.snepslog2))

(def-snepslog.command list-wffs ()
  (mapc #'snepslog-print
	(remove-if-not #'(lambda (node)
			   (node-asupport node))
		       (value.sv 'sneps:nodes))))

(def-snepslog.command list-asserted-wffs
    (&optional (context-name (value.sv 'defaultct)))
  ;; List wffs that are asserted in a context named CONTEXT-NAME.
  (mapc #'snepslog-print
	(remove-if
	 #'(lambda (node)
	     (let ((sneps:crntct context-name))
	       (declare (special sneps:crntct))
	       (or (null (node-asupport node))
		   (not (sneps:isassert.n node)))))
	 (value.sv 'sneps:nodes))))

;(def-snepslog.command end ()     ;used with the old contradiction handler
;  (throw :snepslog-read '(end)))



;;; Added scs/flj 8/1/04
;;; New SNePSLOG commands.
(def-snepslog.command activate! (action &rest args)
  ;; Assertrs and performs the SNePSUL command activate on one wff.
  (mapc #'snepslog-print
	(topsneval
	 (list 'snip:activate
	       (if args
		   (parser (list (cons action args)) 'p (the-empty-frame) 0)
		 (first (parser (list (list action))
				'p (the-empty-frame) 0)))))))

(def-snepslog.command activate (action &rest args)
  ;; Performs the SNePSUL command activate on one term or wff,
  ;;    without asserting it.
  (mapc #'snepslog-print
	(topsneval
	 (list 'snip:activate
	       (if args
		   (cons 'sneps:build
			 (rest
			  (parser (list (list action args))
				  'p (the-empty-frame) 0)))
		 (first (parser (list (list action))
				'p (the-empty-frame) 0)))))))


