;;; -*- Mode: Lisp; Syntax: Common-Lisp; Package: SNEPSLOG; Base: 10 -*-

;; Copyright (C) 1984--2004 Research Foundation of 
;;                          State University of New York

;; Version: $Id: mode3.lisp,v 1.1 2004/08/26 22:03:38 snwiz Exp $

;; This file is part of SNePS.

;; SNePS is free software; you may redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; SNePS is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with SNePS; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA, or to
;; Dr. Stuart C. Shapiro, Department of Computer Science and Engineering,
;; University at Buffalo, The State University of New York, 
;; 201 Bell Hall, Buffalo, NY 14260, USA


(in-package :snepslog)  ;;; added ":"  FLJ 9/2/02

(defconstant *PROPARGS* (make-hash-table :test #'eql)
  "Key is pred; value is (list-of-arcs)")

(defconstant *ARGSPROP* (make-hash-table :test #'equal)
  "Key is (ordered-list-of-arcs); value is arc to pred or (pred)")

(defvar *SILENT-MODE-3* nil
  "If T, mode-3 feedback isn't given.")

(eval-when (load eval)
  (def-snepslog.command set-mode-3 (&optional (silent nil))
    (declare (special *SILENT-MODE-3*))
    (resetnet t)
    (setf *SILENT-MODE-3* silent)
    (make.snepslog3)
    (unless *SILENT-MODE-3*
      (format outunit "~%In SNePSLOG Mode 3.~%Use define-frame <pred> <list-of-arc-labels>.~%~%"))
    (tell "define-frame achieve (action object1)")
    (tell "define-frame ActPlan (nil act plan)")
    (tell "define-frame believe (action object1)")
    (tell "define-frame disbelieve (action object1)")
    (tell "define-frame do-all (action object1)")
    (tell "define-frame do-one (action object1)")
    (tell "define-frame Effect (nil act effect)")
    (tell "define-frame else (nil else)")
    (tell "define-frame GoalPlan (nil goal plan)")
    (tell "define-frame if (nil condition then)")
    (tell "define-frame ifdo (nil if do)")
    (tell "define-frame Precondition (nil act precondition)")
    (tell "define-frame snif (action object1)")
    (tell "define-frame sniterate (action object1)")
    (tell "define-frame snsequence (action object1 object2)")
    (tell "define-frame whendo (nil when do)")
    (tell "define-frame wheneverdo (nil whenever do)")
    (tell "define-frame withall (action vars suchthat do else)")
    (tell "define-frame withsome (action vars suchthat do else)")
    ))

(def-snepslog.command define-frame (pred list-of-args)
  "The first on the LIST-OF-ARGS is the arc to PRED (NIL if ignored).
   The rest on LIST-OF-ARGS are the arcs for the arguments, in order."
  (declare (special *PROPARGS* *ARGSPROP* *SILENT-MODE-3*))
  (check-type pred symbol)
  (check-type list-of-args list)
  (assert (every #'symbolp list-of-args) (list-of-args)
    "The list of arc labels must be an unquoted list of symbols.")
  (mapc #'define-if-not-yet-defined (remove-if #'null list-of-args))
  (when (gethash pred *PROPARGS*)
    (error "~A is already associated with a case-frame." pred))
  (let ((ordered-arcs (sort (remove-if #'null list-of-args) #'sneps::isless.r))
	(predchoice (if (null (first list-of-args)) (list pred)
		      (first list-of-args))))
    (when (and (gethash ordered-arcs *ARGSPROP*)
	       (not (equal (gethash ordered-arcs *ARGSPROP*) predchoice)))
      (error
       "That case-frame is already associated with ~:[the relation~;the predicate~] ~A."
	     (consp (gethash ordered-arcs *ARGSPROP*))
	     (gethash ordered-arcs *ARGSPROP*)))
    (setf (gethash pred *PROPARGS*) list-of-args
	  (gethash ordered-arcs *ARGSPROP*) predchoice))
    (unless *SILENT-MODE-3*
      (feedback pred list-of-args)))

(defun make.snepslog3 ()
  "changes the way snepslog relations are represented in sneps.
   Makes the arcs to the arguments be indexed on the predicate"
  ;; Initialize hash tables to predefined case-frames
  (clrhash *PROPARGS*)
  (clrhash *ARGSPROP*)
  ;; Set Mode 3 functions
  (setf (get 'mode 'relation-argument.list) #'relation-argument.list_mode.3
	(get 'mode 'relation-predicate) #'relation-predicate_mode.3
	(get 'mode 'make-relation) #'make-relation.3))

(defun make-relation.3 (action relation arguments)
  "Receives an action, a relation, and the arguments of the relation.
   Returns the SNePSUL command that perform the action on the relation,
   following the protocol of mode 3."
  (declare (special *PROPARGS*))
  (cons action
	(mapcan #'(lambda (arc arg)
		    (when arc (list arc arg)))
		(gethash relation *PROPARGS*)
		(cons relation arguments))))

(defun relation-predicate_mode.3 (node)
  "Given a sneps node as argument (it should be a relation node),
   returns the relation predicate.
   Works only in snepslog version 3."
  (multiple-value-bind (predchoice foundp)
      (gethash (node-caseframe node) *ARGSPROP*)
    (when foundp
      (if (consp predchoice) (first predchoice)
	(sneps:node-na (sneps:choose.ns (sneps:nodeset.n node predchoice)))))))

(defun relation-argument.list_mode.3 (node)
  "Given a sneps node as argument (it should be a relation node),
   returns the relation arguments.
   Works only in snepslog version 3."
  (mapcar #'(lambda (arc) (sneps:nodeset.n node arc))
	  (rest (gethash (relation-predicate_mode.3 node) *PROPARGS*))))

(defun node-caseframe (node)
  (let (caseframe)
    (do* ((cs (sneps:node-fcableset node) (cddr cs))
	  (arc (first cs) (first cs)))
	((null cs) (nreverse caseframe))
      (when (sneps:isdn.r arc) (push arc caseframe)))))

(defun get-snepslog-version (node)
  (if (typep node 'node)
      (let ((node-snepslog (sneps:node-snepslog node)))
	(if (listp node-snepslog)
	    (getf node-snepslog 'snepslog-version)
	  node-snepslog))
    node))

(defun feedback (pred list-of-args)
  (format outunit "~&~A" pred)
  (showargs (1- (length list-of-args)))
  (format outunit " will be represented by {")
  (showframe pred list-of-args)
  (format outunit "}~%"))

(defun showargs (len)
  (format outunit "(x1")
  (dotimes (i (1- len))
    (format outunit ", x~D"  (+ i 2)))
  (format outunit ")"))

(defun showframe (pred arcs)
  (if (first arcs)
      (format outunit "<~A, ~A>, <~A, x1>" (first arcs) pred (second arcs))
    (format outunit "<~A, x1>" (second arcs)))
  (let ((i 2))
    (dolist (arc (cddr arcs))
      (format outunit ", <~A, x~D>" arc i)
      (incf i))))


    
    




