;;; -*- Mode: Lisp; Syntax: Common-Lisp; Package: SNEPSLOG; Base: 10 -*-

;; Copyright (C) 1984--2004
;; Research Foundation of State University of New York

;; Version: $Id: recognizer.lisp,v 1.12 2004/08/26 23:26:00 snwiz Exp $

;; This file is part of SNePS.

;; SNePS is free software; you may redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; SNePS is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with SNePS; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA, or to
;; Dr. Stuart C. Shapiro, Department of Computer Science and Engineering,
;; University at Buffalo, The State University of New York, 
;; 201 Bell Hall, Buffalo, NY 14260, USA

;; altered for ACL 6 compatibility (FLJ)
		   

(in-package :snepslog)


; Modifications:
;   Choi 2/13/92: Included handling of numerical quantifiers
;   Choi 4/28/92: Included handling of set arguments

; some auxiliar functions

(defun rdc (list)
"Returns a list containig all but the last element of list"
  (if (null (cdr list))
      ()
      (cons (car list) (rdc (cdr list)))))


(defun comma? (exp)
  (eq exp 'snepsul:\,))
(defun make.top.level.list (l)
  (cond ((and (listp (first l)) (null (rest l)) l)
	 (make.top.level.list (first l)))
	((listp l) l)
	(t (list l))))

(defun intercalate (l1 l2)
  (if (null l1)
      l2
      (cons (car l1) (intercalate l2 (cdr l1)))))


(defun ntuple-of-...? (ntuple function separator n)
  (or (and (null ntuple) (zerop n))
      (and (funcall function (first ntuple))
	   (or (= 1 n) (eq separator (second ntuple)))
	   (ntuple-of-...? (cddr ntuple) function separator (1- n)))))

(defun separate (list separator)
  (separate.a.top.level.list list separator))

(defun separate.a.top.level.list (list separator)
  (if (null list)
      nil
      (cons (make.top.level.list (separate.first.element list separator))
	    (separate.a.top.level.list (separate.rest.elements list separator) separator))))

(defun separate.first.element (list separator &aux result)
  (dolist (element list (reverse result))
    (if (eq element separator)
	(return (reverse result))
	(setq result (cons element result)))))

(defun separate.rest.elements (list separator)
  (rest (member separator list :test #'eq)))



(defun list.of.snepslog.expressions? (list &optional (separator 'snepsul:\,))
  (if (not (listp list))
      (return-from list.of.snepslog.expressions? nil))
  (setf list (separate list separator))
  (dolist (element list t)
    (unless (snepslog-expression? element)
      (return nil))))



(defun wff->m (wff)
  (intern  
   (build-namestring :m (string-trim "WFwf" (symbol-name (first (flistify wff)))))
   'snepsul))


(defun wffs->ms (list)
  (mapcar #'wff->m
	  (remove-if #'(lambda (x) (eq x 'snepsul:\,))
		  list)))



(defun remove-context-and-command (SNePSLog.exp)
"Receives as argument a SNePSLog expression. Returns the SNePSLog
expression without the context and command specification"
  (remove-command (remove-context SNePSLog.exp)))

(defun remove-context (SNePSLog.exp)
"Receives as argument a SNePSLog expression. Returns the SNePSLog
expression without the context specification"
  (if (eql (second SNePSLog.exp) 'snepsul:\:)
      (cddr SNePSLog.exp)
      SNePSLog.exp))

(defun remove-command (SNePSLog.exp)
"Receives as argument a SNePSLog expression. Returns the SNePSLog
expression without the command specification"
  (if (member (first (last SNePSLog.exp))
	      '(snepsul:|.| snepsul:|!| snepsul:|?| snepsul:|??|)
	      :test #'eq)
      (rdc SNePSLog.exp)
      snepslog.exp))

;; Modified to handle deduce commands with option, e.g. `a(b,c)?(1 0)'
(defun get-command (SNePSLog.exp)
  ;; Returns the command of SNePSLog.exp.
  ;; If there is none specified, the default is `snepsul:.', i.e., assert.
  (let ((possibly-a-command (car (last SNePSLog.exp)))
	(len (length SNePSLog.exp)))
    (cond ((member possibly-a-command '(snepsul:|.| snepsul:|!| snepsul:|?|
					snepsul:|??|) :test #'eq)
	   possibly-a-command)
	  ((and (> len 1)
		(eq 'snepsul:|?| (nth (- len 2) SNePSLog.exp)))
	   'snepsul:|?|)
	  (t 'snepsul:\.))))

(defun snepslog-deduce-with-option? (exp)
  ;; Returns T if next-to-last element of EXP is `?'.
  (let ((len (length exp)))
    (eq (nth (- len 2) exp) 'snepsul::|?|)))

(defun remove-option-from-deduce (exp)
  ;; Removes number-of-answers option from a deduce command supplied in EXP.
  ;; E.g., from `a(b,c)?(1 0)' it returns `a(b,c)'.
  (butlast (butlast exp)))

(defun get-snepslog-context (SNePSLog.exp)
  "Receives as argument a SNePSLog expression. Returns the SNePSLog
expression's context. If there is none specified, returns nothing"
  (when (and (eql (second SNePSLog.exp) 'snepsul:\:)
	     (symbolp (first SNePSLog.exp)))
    `(:context ,(first SNePSLog.exp))))


(defun sneps-action (command)
"Receives as argument a SNePSLog command. Returns the correspondent SNePS command"
  (cond ((eq command 'snepsul:\.) 'sneps:assert)
	((eq command 'snepsul:\?) 'snip:deduce)
	((eq command 'snepsul:\!) 'snip:add)
	((eq command 'snepsul:\??) 'sneps:findassert)))

(defun auxiliar-action (action)
"Receives as argument a sneps action. Returns the auxiliar action."
  (if (not (member action '(sneps:findassert sneps:find)))
      'sneps:build
      'sneps:find))






(defun no-variables () nil)

(defun is-variable? (SNePSLog.exp Var.stack)
"Var.stack is a stack that contains the variables"
  (or (member (first (flistify SNePSLog.exp)) (first var.stack) :test #'eq)
      (and var.stack (is-variable? snepslog.exp (rest var.stack)))))

(defun add-to-variables (var.list var.stack)
  (cons (append var.list (first var.stack))
	var.stack))






(defun quantifier-variables-list (quantifier-and-variables)
  (remove-if #'(lambda (x) (eq x 'snepsul:\,))
	  (rest quantifier-and-variables)))

(defun quantifier (quantifier-and-variables)
  (car quantifier-and-variables))

(defparameter *registers* nil)

(defun convert-to-sneps-variable (snaction var)
"Receives as arguments a sneps action and a snepslog variable. Returns the adequate sneps variable"
  (declare (special registers)) ;registers is the variable that contains the atn-registers.
  (cond ((member snaction '(sneps:find sneps:findassert))
	 `(sneps:\? ,(car (flistify var))))
	((member snaction '(sneps:build snip:deduce) :test #'eq)
	 (if (member (car (flistify var)) *registers* :test #'eq)
	     `(sneps:\* ',(car (flistify var)))
	     (progn 
	       (setf *registers* (cons (car (flistify var)) *registers*))
	       `(sneps:\$ ',(car (flistify var))))))
	(t var)))


;
; the following functions are used to implement the non.predicate feature. It is used to declare that some atoms are not
; to be understood as predicates. The negation is one example. ~(A) is not to be considered as the predicate ~ and with
; the  argument A
;


(defun declare-non.predicate (exp)
  (setf (get 'non.predicate exp) t))

(defun non.predicate? (exp)
  (get 'non.predicate exp))

(declare-non.predicate 'snepsul:~)


;
; The following functions are used to deal with relations.
;

(defun relation-arguments (snepslog.relation)
  "Given a relation as arguments, returns the arguments of that relation.
   For instance, given the relation a(b,c) - the lisp representation for
   this relation is the list (a (b |,| c)) - returns the list (b c)."
  (separate (second snepslog.relation) 'snepsul:|,|))


(defun relation (snepslog.relation)
"Given a relation as arguments, returns the arguments of that relation. For instance, given the relation a(b,c) - the lisp
representation for this relation is the list (a (b |,|  c)) - returns the atom a."
  (first snepslog.relation))



(defun associate-arcs (arc arcs)
"Used in version 2. Given the predicate arc, returns the other arcs of the node"
  (setf (get arc 'snepslog-associated-arcs) arcs))

(defun define-if-not-yet-defined (arc)
  "Receives as argument an atom, arc. If already exists an arc with that name, that arc is returned. Otherwise, creates and
 returns a new arc with that name"
  (if (not (sneps:is.r arc))
      (sneps:new.r arc)))

(defun make-relation.1 (action relation arguments)
  "Receives an action, a relation and the arguments of the relation. Returns the SNePSUL command that perform the action
on the relation (following the protocol of mode 1, that is, the predicate arc named is r and the argument arcs are a1, a2,
 etc.)."  
  (let ((counter 0))
    (define-if-not-yet-defined 'snepsul:r)
    (nconc (list action 'snepsul:r relation)
	   (mapcan #'(lambda (x)
		       (list 
			(let ((arc (intern 
				    (build-namestring :a (incf counter))
				    'snepsul)))
			  (define-if-not-yet-defined arc)
			  arc)
			x)) 
		   arguments))))

(defun make-relation.2 (action relation arguments)
  "Receives an action, a relation and the arguments of the relation. Returns the SNePSUL command that perform the action
on the relation (following the protocol of mode 2, that is, the predicate arc named is indexed with the predicate name and
 the argument arcs are also indexed with the predicate. The predicate arc name is the result of appending the string ' rel '
 to the predicate name The first white space is to garantee that the predicate has the lowest alfabetic value of all the arcs
leaving any sneps node. This way, it is easy and some how inexpensive to find the the predicate of a sneps node. The arguments arc is the result of appending the string 'rel-arg ' to the predicate name and a counter. The argument arcs are associated
to the predicate)."
  (let ((counter 0)
	(relation-arc-name 
	 (build-namestring " " :rel " " (symbol-name relation)))
	(arguments-prefix  
	 (build-namestring  :rel-arg# relation))
	argument-arcs
	relation-arc)
    (setq argument-arcs 
      (mapcar #'(lambda (node)
		  (declare (ignore node))
		  (intern 
		   (build-namestring arguments-prefix (incf counter))
		   'snepsul))
	      arguments)
      relation-arc (intern relation-arc-name 'snepsul))  
    (mapc #'define-if-not-yet-defined (cons relation-arc argument-arcs))
    (associate-arcs relation-arc argument-arcs)
    (cons action (intercalate (cons relation-arc argument-arcs)
			      (cons relation arguments)))))

(defun make-relation (action relation arguments)
  (funcall (get 'mode 'make-relation) action relation arguments))




(defun snepslog-and-or-arguments (and-or exp)
  (separate exp and-or))

(defun snepslog-expression-quantifier (exp)
  (cons (first exp) (second exp)))


(defun snepsul-quantifier (q)
  (cond ((eq (first q) 'snepsul:all)
	 (cons 'sneps:forall (rest q)))
	((eq (first q) 'snepsul:exists)
	 (cons 'sneps:exists (rest q)))
	(t
	 (cons 'sneps:pevb (rest q)))))

(defun snepslog-antecedent-list (exp)
  (if (member 'snepsul:\=> exp :test #'eq)
      (list (make.top.level.list (separate.first.element exp 'snepsul:=>)))
      (separate (first exp) 'snepsul:\,)))

(defun snepslog-consequent-list (exp)
  (let ((result (rest (member 'snepsul:\=> exp :test #'eq))))
    (if result ;then it's a simple entailment!
	(list (make.top.level.list result))
	(separate (third exp) 'snepsul:\,))))


(defun snepslog-threshed-propositions (exp)
  (separate (third exp) 'snepsul:\,))

(defun get-thresh-i (exp)
  (caadr exp))

(defun get-thresh-j (exp)
  (caddr(cadr exp)))

(defun snepslog-andor-arguments (exp)
  (separate (third exp) 'snepsul:\,))

(defun get-min (exp)
  (first (second exp)))

(defun get-max (exp)
  (third (second exp)))

(defun get-quantified-expression (exp)
  (or (caddr (member 'snepsul:all exp :test #'eq))
      (caddr (member 'snepsul:exists exp :test #'eq))))

(defun put-quantifier-in-expression (quantifier-and-variables expression)
  (if (or (member 'sneps:forall expression :test #'eq)
	  (member 'sneps:exists expression :test #'eq))
      (append (list (first expression) ;snaction
		    (quantifier quantifier-and-variables)
		    (mapcar #'(lambda (x) `(sneps:$ ',x))
			    (quantifier-variables-list quantifier-and-variables))
		    'min 1 'max 1
		    'sneps:arg)
	      (cons (cons (auxiliar-action (first expression))
			  (rest expression))
		    nil))
      (append (list (first expression) 
		    (quantifier quantifier-and-variables) 
		    (mapcar #'(lambda (x) `(sneps:$ ',x))
			    (quantifier-variables-list quantifier-and-variables)))
	      (rest expression))))


;; Numerical quantifier stuff:
;; from nexists(2,4,5)(x,y) ({person(x),dog(y),owns(x,y)}:{spoils(x,y)}),
;; returns (nexists (x,y)).

(defun snepslog-expression-num-quantifier (exp)
  (cons (first exp) (third exp)))

;; from nexists(2,4,5)(x,y) ({person(x),dog(y),owns(x,y)}:{spoils(x,y)}),
;; returns (person(x) dog(y) owns(x,y)).

(defun snepslog-num-quant-ant-list (exp)
  (if (member 'snepsul:\: exp :test #'eq)
      (separate (first exp) 'snepsul:\,)))

;; from nexists(2,4,5)(x,y) ({person(x),dog(y),owns(x,y)}:{spoils(x,y)}),
;; returns (spoils(x,y)).

(defun snepslog-num-quant-cq-list (exp)
  (if (member 'snepsul:\: exp :test #'eq)
      (separate (third exp) 'snepsul:\,)))

;; get the value of each parameter emin, emax, etot 
;; in a numerical quantifier parameter expression
;;
;; if a parameter value is a number, return parameter name and value.
;; if it is '_', nothing is included for that parameter.
;;
;;  e.g., from nexists(2,4,5), returns (etot 5 emin 2 emax 4)
;;        from nexists(3,_,5), returns (etot 5 emin 3)
;;        from nexists(_,1,_), returns (emax 1)

(defun get-emin-emax-etot (exp)
  (let ((emin-value (get-emin exp))
	(emax-value (get-emax exp))
	(etot-value (get-etot exp)))
    (append (if (numberp emin-value)
		(list 'emin emin-value))
	    (if (numberp emax-value)
		(list 'emax emax-value))
	    (if (numberp etot-value)
		(list 'etot etot-value)))))

(defun get-emin (exp)
  (first (second exp)))

(defun get-emax (exp)
  (third (second exp)))

(defun get-etot (exp)
  (fifth (second exp)))

(defun get-num-quantified-expression (exp)
  (cadddr (member 'snepsul:nexists exp :test #'eq)))

(defun put-num-quantifier-in-expression (quantifier-and-variables expression)
  (append (list (quantifier quantifier-and-variables) 
		(mapcar #'(lambda (x) `(sneps:$ ',x))
			(quantifier-variables-list quantifier-and-variables)))
	  (rest expression)))



    
    




