;;; -*- Mode: Lisp; Syntax: Common-Lisp; Package: SNEPSLOG; Base: 10 -*-

;; Copyright (C) 1984--2004
;; Research Foundation of State University of New York

;; Version: $Id: tellask.lisp,v 1.6 2004/08/26 23:25:51 snwiz Exp $

;; This file is part of SNePS.

;; SNePS is free software; you may redistribute it and/or modify
;; it under the terms of the GNU General Public License as published
;; by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; SNePS is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with SNePS; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139,
;; USA, or to
;; Dr. Stuart C. Shapiro, Department of Computer Science, State
;; University of
;; University at Buffalo, The State University of New York, 
;; 201 Bell Hall, Buffalo, NY 14260, USA


;;; Tell-Ask Interface to SnePS
;;; by Stuart C. Shapiro
;;; August 12, 1997

;;; UPDATE by scs, 8/26/2004  (flj)
;;; Fixes bugs in the tell-ask interface code,
;;;    and provides ask, askifnot, askwh, and askwhnot as SNePSLOG commands.


(in-package :snepslog) 

(defun tell (&rest strings)
  "Strings must be valid SNePSLOG inputs.
   Gives each string to the SNePSLOG interpreter.
   Prints what SNePSLOG would print, and returns NIL."
  (sneps:in.environment
   :variables ((*package* (find-package 'snepsul))
	       (*print-length* nil)
	       (*print-level* nil)
	       (*print-pretty* t)
	       (old-infertrace snip:*infertrace*)
	       (sneps:outunit nil))
   :functions ((surface 'snepslog:surface)
	       (slight-surface #'snepslog:slight-surface)
	       (node-intern #'snepslog:node-intern)
	       (sneps-node? #'snepslog:sneps-node?))
   :eval (progn
	   (sneps::snepslog-init "")
	   (let ((sneps:outunit t)
		 (sneps:inunit t))
	     (dolist (string strings)
	       (catch 'sneps:sneps-error
		 (snepslog:snepslog-print
		  (sneps:topsneval
		   (snepslog::snepslog-read-from-string 
		    (snepslog::change.brackets! string)))
		  t)))))
   :always.do.this (progn (setq snip:*infertrace* old-infertrace)
			  (snepslog:snepslogreadoff)
			  nil)))

(defun ask (string &key verbose)
  "String must be a valid SNePSLOG input.
   Gives that string to the SNePSLOG interpreter.
   but uses DEDUCETRUE instead of DEDUCE.
   Returns what the SNePSUL interpreter would if given the SNePSUL
   version of the string.
   If :VERBOSE is T, prints the results as well as returning it."
  (sneps:in.environment
   :variables ((*package* (find-package 'snepsul))
	       (*print-length* nil)
	       (*print-level* nil)
	       (*print-pretty* t)
	       (old-infertrace snip:*infertrace*)
	       (sneps:outunit nil))
   :functions ((surface 'snepslog:surface)
	       (slight-surface #'snepslog:slight-surface)
	       (node-intern #'snepslog:node-intern)
	       (sneps-node? #'snepslog:sneps-node?))
   :eval (progn
	   (sneps::snepslog-init "") 
	   (catch 'sneps:sneps-error
	     (let* ((sneps:outunit t)
		    (sneps:inunit t)
		    (command
		     (snepslog::snepslog-read-from-string 
		      (if (member (char string (1- (length string)))
				  '(#\. #\! #\?) :test #'char=)
			  string (concatenate 'string string "?"))))
		    (results
		     (sneps:topsneval
		      (if (eql (first command) 'snip:deduce)
			  (cons 'snip:deducetrue (rest command))
			command))))
	       (when verbose (mapc #'snepslog:snepslog-print results))
	       results)))
   :always.do.this (progn (setq snip:*infertrace* old-infertrace)
			  (snepslog:snepslogreadoff)
			  nil)))

(defun askifnot (string &key verbose)
  "String must be a valid SNePSLOG input.
   Gives that string to the SNePSLOG interpreter.
   but uses DEDUCEFALSE instead of DEDUCE.
   Returns what the SNePSUL interpreter would if given the SNePSUL
   version of the string.
   If :VERBOSE is T, prints the results as well as returning it."
  (sneps:in.environment
   :variables ((*package* (find-package 'snepsul))
	       (*print-length* nil)
	       (*print-level* nil)
	       (*print-pretty* t)
	       (old-infertrace snip:*infertrace*)
	       (sneps:outunit nil))
   :functions ((surface 'snepslog:surface)
	       (slight-surface #'snepslog:slight-surface)
	       (node-intern #'snepslog:node-intern)
	       (sneps-node? #'snepslog:sneps-node?))
   :eval (progn
	   (sneps::snepslog-init "") 
	   (catch 'sneps:sneps-error
	     (let* ((sneps:outunit t)
		    (sneps:inunit t)
		    (command
		     (snepslog::snepslog-read-from-string
		      (if (member (char string (1- (length string)))
				  '(#\. #\! #\?) :test #'char=)
			  string (concatenate 'string string "?"))))
		    (results
		     (sneps:topsneval
		      (if (eql (first command) 'snip:deduce)
			  (cons 'snip:deducefalse (rest command))
			command))))
	       (when verbose (mapc #'snepslog:snepslog-print results))
	       results)))
   :always.do.this (progn (setq snip:*infertrace* old-infertrace)
			  (snepslog:snepslogreadoff)
			  nil)))

(defun askwh (string &key verbose)
  "String must be a valid SNePSLOG input.
   Gives that string to the SNePSLOG interpreter,
   but uses DEDUCEWH instead of DEDUCE.
   Returns what the SNePSUL interpreter would if given the SNePSUL
   version of the string.
   If :VERBOSE is T, prints the results as well as returning it."
  (sneps:in.environment
   :variables ((*package* (find-package 'snepsul))
	       (*print-length* nil)
	       (*print-level* nil)
	       (*print-pretty* t)
	       (old-infertrace snip:*infertrace*)
	       (sneps:outunit nil))
   :functions ((surface 'snepslog:surface)
	       (slight-surface #'snepslog:slight-surface)
	       (node-intern #'snepslog:node-intern)
	       (sneps-node? #'snepslog:sneps-node?))
   :eval (progn
	   (sneps::snepslog-init "") 
	   (catch 'sneps:sneps-error
	     (let* ((sneps:outunit t)
		    (sneps:inunit t)
		    (command
		     (snepslog::snepslog-read-from-string
		      (if (member (char string (1- (length string)))
				  '(#\. #\! #\?) :test #'char=)
			  string (concatenate 'string string "?"))))
		    (results
		     (sneps:topsneval
		      (cond ((eql (first command) 'snip:deduce)
			     (cons 'snip:deducewh (rest command)))
			    ((eql (first command) 'findassert)
			     (cons 'snip:deducewh
				   (cons '(0 0)
					 (mapcar
					  #'(lambda (elt)
					      (if (and (consp elt)
						       (eql (first elt)
							    '?))
						  `($ ',(second elt))
						elt))
					  (rest command)))))
			    (t command)))))
	       (when verbose (mapc #'snepslog:snepslog-print results))
	       results)))
   :always.do.this (progn (setq snip:*infertrace* old-infertrace)
			  (snepslog:snepslogreadoff)
			  nil)))

(defun askwhnot (string &key verbose)
  "String must be a valid SNePSLOG input.
   Gives that string to the SNePSLOG interpreter,
   but uses DEDUCEWHNOT instead of DEDUCE.
   Returns what the SNePSUL interpreter would if given the SNePSUL
   version of the string.
   If :VERBOSE is T, prints the results as well as returning it."
  (sneps:in.environment
   :variables ((*package* (find-package 'snepsul))
	       (*print-length* nil)
	       (*print-level* nil)
	       (*print-pretty* t)
	       (old-infertrace snip:*infertrace*)
	       (sneps:outunit nil))
   :functions ((surface 'snepslog:surface)
	       (slight-surface #'snepslog:slight-surface)
	       (node-intern #'snepslog:node-intern)
	       (sneps-node? #'snepslog:sneps-node?))
   :eval (progn
	   (sneps::snepslog-init "") 
	   (catch 'sneps:sneps-error
	     (let* ((sneps:outunit t)
		    (sneps:inunit t)
		    (command
		     (snepslog::snepslog-read-from-string
		      (if (member (char string (1- (length string)))
				  '(#\. #\! #\?) :test #'char=)
			  string (concatenate 'string string "?"))))
		    (results
		     (sneps:topsneval
		      (cond ((eql (first command) 'snip:deduce)
			     (cons 'snip:deducewhnot (rest command)))
			    ((eql (first command) 'sneps:findassert)
			     (cons 'snip:deducewhnot
				   (cons '(0 0)
					 (mapcar
					  #'(lambda (elt)
					      (if (and (consp elt)
						       (eql (first elt)
							    '?))
						  `($ ',(second elt))
						elt))
					  (rest command)))))
			    (t command)))))
	       (when verbose (mapc #'snepslog:snepslog-print results))
	       results)))
   :always.do.this (progn (setq snip:*infertrace* old-infertrace)
			  (snepslog:snepslogreadoff)
			  nil)))
;;;
;;; SNePSLOG command versions of the ask functions.
;;;
(def-snepslog.command ask (action &rest args)
  "Does ask on the argument."
  (ask (format nil "~A~{~A~}?" action args) :verbose t)
  (values))

(def-snepslog.command askifnot (action &rest args)
  "Does askifnot on the argument."
  (askifnot (format nil "~A~{~A~}?" action args) :verbose t)
  (values))

(def-snepslog.command askwh (action &rest args)
  "Does askwh on the argument."
  (askwh (format nil "~A~{~A~}?" action args) :verbose t)
  (values))

(def-snepslog.command askwhnot (action &rest args)
  "Does askwhnot on the argument."
  (askwhnot (format nil "~A~{~A~}?" action args) :verbose t)
  (values))
