;;; -*- Mode: Lisp; Syntax: Common-Lisp; Package: SNIP; Base: 10 -*-

;; Copyright (C) 1984--2004 Research Foundation of 
;;                          State University of New York

;; Version: $Id: channel.lisp,v 1.6 2004/08/26 23:26:07 snwiz Exp $

;; This file is part of SNePS.

;; SNePS is free software; you may redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; SNePS is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with SNePS; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA, or to
;; Dr. Stuart C. Shapiro, Department of Computer Science and Engineering,
;; University at Buffalo, The State University of New York, 
;; 201 Bell Hall, Buffalo, NY 14260, USA

(in-package :snip)


; =============================================================================
;
; <channel>  ::=  ( <filter> <switch> <context> <destination> <valve> )
;
; -----------------------------------------------------------------------------
;
; RECOGNIZERS    is.ch      :  <universal> --> <boolean>
;
; CONSTRUCTORS   make.ch    :  <filter> x <switch> x <context> x
;                                   <destination> x <valve> --> <channel>
;
; SELECTORS      filter.ch      : <channel> --> <filter>
;                switch.ch      : <channel> --> <switch>
;                context.ch     : <channel> --> <context>
;                destination.ch : <channel> --> <destination>
;                valve.ch       : <channel> --> <valve>
;
; TESTS          isopen.ch      : <channel> --> <boolean>
;                isclosed.ch    : <channel> --> <boolean>
;                equivalent.ch  : <channel> x <channel> --> <boolean>
;
; UTILITY        open.ch      : <channel> -->
;                close.ch     : <channel> -->
;
; =============================================================================



;
; =============================================================================
;
; make.ch
; -------
;
;       arguments     : f - <filter>
;                       s - <switch>
;                       c - <context>
;                       d - <destination>
;                       v - <valve>
;
;       returns       : <channel>
;
;       description   : returns the <channel> consisting of the components
;                        passed as arguments
;
;                                        written :  rgh 07/29/85
;                                        modified:  rgh 08/19/85
;                                        modified:  cpf 10/06/88
;
(defmacro make.ch (f s c d v)
  `(list ,f ,s ,c ,d ,v))
;
;
; =============================================================================
;
; filter.ch
; ---------
;
;       arguments     : channel  -  <channel>
;
;       returns       : <filter>
;
;       description   : selects the filter of the channel
;
;                                        written :  rgh 05/09/85
;                                        modified:  rgh 06/11/85
;
;
(defmacro filter.ch (channel)
   `(first ,channel))
;
;
; =============================================================================
;
; switch.ch
; ---------
;
;       arguments     : channel  -  <channel>
;
;       returns       : <switch>
;
;       description   : selects the switch of the channel
;
;                                        written :  rgh 05/09/85
;                                        modified:  rgh 06/11/85
;
;
(defmacro switch.ch (channel)
   `(second ,channel))
;
; =============================================================================
;
; context.ch
; ----------
;
;       arguments     : channel  -  <channel>
;
;       returns       : <context>
;
;       description   : selects the context of the channel
;
;                                        written :  cpf 10/06/88
;                                        modified:  
;
;
(defmacro context.ch (channel)
   `(third ,channel))
;
;
; =============================================================================
;
; destination.ch
; --------------
;
;       arguments     : channel  -  <channel>
;
;       returns       : <destination>
;
;       description   : selects the destination of the channel
;
;                                        written :  rgh 05/09/85
;                                        modified:  rgh 08/18/85
;                                        modified:  cpf 10/06/88
;
(defmacro destination.ch (channel)
   `(fourth ,channel))
;
;
; =============================================================================
;
; valve.ch
; --------
;
;       arguments     : channel  -  <channel>
;
;       returns       : <valve>
;
;       description   : selects the valve of the channel
;
;                                        written :  rgh 08/19/85
;                                        modified:  cpf 10/06/88
;
;
(defmacro valve.ch (channel)
   `(fifth ,channel))
;
;
; =============================================================================
;
; is.ch
; -----
;
;       arguments     : u - <universal>
;
;       returns       : <boolean>
;
;       description   : returns "true" if "u" is a <channel>,
;                               "false" otherwise
;
;                                        written :  rgh  7/29/85
;                                        modified:  rgh  8/19/85
;                                                   rgh  3/08/86
;                                                   cpf 10/06/88
(defmacro is.ch (u)
  `(and (listp ,u)
        (is.filter (filter.ch ,u))
        (is.switch (switch.ch ,u))
	(sneps:is.ct (context.ch ,u))
        (is.dest (destination.ch ,u))
        (is.valve (valve.ch ,u))))
;
;
; =============================================================================
;
; isopen.ch
; ---------
;
;       arguments     : channel - <channel>
;
;       returns       : <boolean>
;
;       description   : returns "true" if the <valve> of "channel" is open
;
;                                        written :  rgh 08/19/85
;                                        modified:
;
;
(defmacro isopen.ch (channel)
  `(eq (valve.ch ,channel) 'OPEN))
;
;
; =============================================================================
;
; isclosed.ch
; -----------
;
;       arguments     : channel - <channel>
;
;       returns       : <boolean>
;
;       description   : returns "true" if the <valve> of "channel" is closed
;
;                                        written :  rgh 08/19/85
;                                        modified:
;
;
(defmacro isclosed.ch (channel)
  `(eq (valve.ch ,channel) 'CLOSED))
;
;
; =============================================================================
;
; equivalent.ch
; -------------
;
;       arguments     : ch1 - <channel>
;                       ch2 - <channel>
;
;       returns       : <boolean>
;
;       description   : returns "true" if "ch1" and "ch2" are equivalent
;                       in that their destinations and contexts are equal,
;                       and their filters and switches define equivalent
;                       restrictions.
;                       Note that their valves are not tested, so that a
;                       closed channel may match an otherwise equivalent
;                       open one.
;
;                                        written :  rgh 10/06/85
;                                        modified:  rgh  3/22/86
;                                        modified:  cpf 10/06/88
;
(defmacro equivalent.ch (ch1 ch2)
  `(and (iseq.dest (destination.ch ,ch1) (destination.ch ,ch2))
        (equivalent.restr
             (make.restr (filter.ch ,ch1))
             (make.restr (filter.ch ,ch2)))
        (equivalent.restr
             (make.restr (switch.ch ,ch1))
             (make.restr (switch.ch ,ch2)))
	(sneps:iseq.ct (context.ch ,ch1) (context.ch ,ch2))))
;
;
; =============================================================================
;
; open.ch
; -------
;
;       arguments     : channel - <channel>
;
;       description   : destructively modifies "channel" so that its <valve>
;                       is set to OPEN
;
;                                        written :  rgh 08/19/85
;                                        modified:  rgh 11/30/85
;                                        modified:  cpf 10/06/88
;
(defmacro open.ch (channel)
  `(rplaca (cdddddr ,channel) 'OPEN))
;
;
; =============================================================================
;
; close.ch
; --------
;
;       arguments     : channel - <channel>
;
;       description   : destructively modifies "channel" so that its <valve>
;                       is set to CLOSED
;
;                                        written :  rgh 08/19/85
;                                        modified:  rgh 11/30/85
;                                        modified:  cpf 10/07/88
;
(defmacro close.ch (channel)
  `(rplaca (cdddddr ,channel) 'CLOSED))
;
;
; =============================================================================



    
    




