;;; -*- Mode: Lisp; Syntax: Common-Lisp; Package: SNIP; Base: 10 -*-

;; Copyright (C) 1984--2004 Research Foundation of 
;;                          State University of New York

;; Version: $Id: cqchset.lisp,v 1.6 2004/08/26 23:26:07 snwiz Exp $

;; This file is part of SNePS.

;; SNePS is free software; you may redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; SNePS is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with SNePS; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA, or to
;; Dr. Stuart C. Shapiro, Department of Computer Science and Engineering,
;; University at Buffalo, The State University of New York, 
;; 201 Bell Hall, Buffalo, NY 14260, USA

(in-package :snip)


; =============================================================================
;
; <cq-channel set> ::= { <cq-channel> ... }
;
; -----------------------------------------------------------------------------
;
; PRIMITIVE      new.cqchset :  --> <cq-channel set>
;  ELEMENTS
;
; RECOGNIZERS    isnew.cqchset : <cq-channel set> --> <boolean>
;
; CONSTRUCTORS   putin.cqchset   : <cq-channel> x <cq-channel set>
;                                     --> <cq-channel set>
;                insert.cqchset  : <cq-channel> x <cq-channel set>
;                                           --> <cq-channel set>
;                makeone.cqchset : <cq-channel> --> <cq-channel set>
;                update.cqchset  : <cq-channel> x <cq-channel set>
;                                           --> <cq-channel set>
;
; SELECTORS      choose.cqchset : <cq-channel set> --> <cq-channel>
;                others.cqchset : <cq-channel set> --> <cq-channel set>
;
; TESTS          ismemb.cqchset : <cq-channel> x <cq-channel set> --> <boolean>
;
; UTILITIES      get-updated-cqch : <cq-channel> x <cq-channel set> --> <cq-channel>
;
; =============================================================================
;
; new.cqchset
; -----------
;
;       returns       : <cq-channel set>
;
;       description   : returns a "new" <cq-channel set>
;
;                                        written :  rgh  2/08/86
;                                        modified:
;
;
(defmacro new.cqchset ()
  `(new.Set))
;
;
; =============================================================================
;
; isnew.cqchset
; -------------
;
;       arguments     : cqchs - <cq-channel set>
;
;       returns       : <boolean>
;
;       description   : returns "true" if "cqchs" is a "new" <cq-channel set>,
;                       "false" otherwise
;
;                                        written :  rgh  2/08/86
;                                        modified:
;
;
(defmacro isnew.cqchset (cqchs)
  `(isnew.Set ,cqchs))
;
;
; =============================================================================
;
; putin.cqchset
; -------------
;
;       arguments     : cqch - <cq-channel>
;                       cqchs - <cq-channel set>
;
;       returns       : <cq-channel set>
;
;       description   : returns a <cq-channel set> consisting of "cqchs" with
;                       "cqch" inserted.  It is assumed that "cqch" was not
;                       already in "cqchs".
;
;                                        written :  rgh  2/08/86
;                                        modified:
;
;
(defmacro putin.cqchset (cqch cqchs)
  `(putin.Set ,cqch ,cqchs))
;
;
; =============================================================================
;
; makeone.cqchset
; ---------------
;
;       arguments     : cqch - <cq-channel>
;
;       returns       : <cq-channel set>
;
;       description   : returns a <cq-channel set> with the single element
;                       "cqch"
;
;                                        written :  rgh  2/12/86
;                                        modified:
;
;
(defmacro makeone.cqchset (cqch)
  `(makeone.Set ,cqch))
;
;
; =============================================================================
;
; choose.cqchset
; --------------
;
;       arguments     : cqchs - <cq-channel set>
;
;       returns       : <cq-channel>
;
;       description   : returns and element of "cqchs"
;
;                                        written :  rgh  2/08/86
;                                        modified:
;
;
(defmacro choose.cqchset (cqchs)
  `(choose.Set ,cqchs))
;
;
; =============================================================================
;
; others.cqchset
; --------------
;
;       arguments     : cqchs - <cq-channel set>
;
;       returns       : <cq-channel set>
;
;       description   : returns "cqchs" with the element chosen by
;                       choose.cqchset eliminated
;
;                                        written :  rgh  2/08/86
;                                        modified:
;
;
(defmacro others.cqchset (cqchs)
  `(others.Set ,cqchs))
;
;
; =============================================================================
;
; insert.cqchset
; --------------
;
;       arguments     : cqch - <cq-channel>
;                       cqchset - <cq-channel set>
;
;       returns       : <cq-channel set>
;
;       description   : returns a <cq-channel set> similar to "cqchset" but
;                       with "cqch" inserted if it was not already there
;
;       implementation: "cqch" is considered to be in "cqchset" already if
;                       there is already a <cq-channel> is "cqchset" with
;                       the same <channel>
;
;                                        written :  rgh  2/12/86
;                                        modified:  rgh  3/30/86
;                                                   rgh  4/03/86
;
;
(defun insert.cqchset (cqch cqchset)
  (cond ((isnew.cqchset cqchset) (makeone.cqchset cqch))
        ((equivalent.ch (channel.cqch cqch)
                        (channel.cqch (choose.cqchset cqchset)))
	 cqchset)
        (t (putin.cqchset (choose.cqchset cqchset)
			  (insert.cqchset cqch (others.cqchset cqchset))))))
;
;
; =============================================================================
;
; update.cqchset
; --------------
;
;       arguments     : cqch - <cq-channel>
;                       cqchset - <cq-channel set>
;
;       returns       : <cq-channel set>
;
;       description   : returns a <cq-channel set> similar to "cqchset" but
;                       with "cqch" replacing a <cq-channel> in "cqchset"
;                       that has an equivalent <channel>, or with "cqch"
;                       inserted if no matching <cq-channel> is found.
;
;                                        written :  rgh  4/03/86
;                                        modified:
;
;
(defun update.cqchset (cqch cqchset)
  (cond ((isnew.cqchset cqchset) (makeone.cqchset cqch))
        ((equivalent.ch (channel.cqch cqch)
			(channel.cqch (choose.cqchset cqchset)))
	 (putin.cqchset cqch (others.cqchset cqchset)))
        (t (putin.cqchset (choose.cqchset cqchset)
			  (update.cqchset cqch (others.cqchset cqchset))))))
;
;
; =============================================================================
;
; ismemb.cqchset
; --------------
;
;       arguments     : cqch - <cq-channel>
;                       cqchs - <cq-channel set>
;
;       returns       : <boolean>
;
;       description   : returns "true" if there is a <cq-channel> in "cqchs"
;                       which has a <channel> equivalent to the <channel> of
;                       "cqch", "false" otherwise
;
;                                        written :  rgh  2/08/86
;                                        modified:
;
;
(defmacro ismemb.cqchset (cqch cqchs)
  `(prog (cqchset)
	 (setq cqchset ,cqchs)
      begin
         (if (isnew.cqchset cqchset) (return nil))
         (if (equivalent.ch (channel.cqch ,cqch)
			    (channel.cqch (choose.cqchset cqchset)))
	     (return t))
	 (setq cqchset (others.cqchset cqchset))
	 (go begin)))
;
;
; =============================================================================
;
; get-updated-cqch
; ----------------
;
;       arguments     : cqch - <cq-channel>
;                       cqchs - <cq-channel set>
;
;       returns       : <cq-channel>
;
;       description   : Returns the <cq-channel> equivalent to 'cqch' that
;                       is present in 'cqchs'
;                       
;                       
;
;                                        written : njm/cpf 12/14/88 
;                                        modified:
;
;
(defun get-updated-cqch (cqch cqchs)
  (let ((result nil))
    (dolist (firstcqch cqchs result)
      (when (equivalent.ch (channel.cqch cqch) (channel.cqch firstcqch))
	(setq result firstcqch)))))






  
;
;
; =============================================================================



    
    




