;;; -*- Mode: Lisp; Syntax: Common-Lisp; Package: SNIP; Base: 10 -*-

;; Copyright (C) 1984--2004 Research Foundation of 
;;                          State University of New York

;; Version: $Id: fnodeset.lisp,v 1.6 2004/08/26 23:26:07 snwiz Exp $

;; This file is part of SNePS.

;; SNePS is free software; you may redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; SNePS is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with SNePS; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA, or to
;; Dr. Stuart C. Shapiro, Department of Computer Science and Engineering,
;; University at Buffalo, The State University of New York, 
;; 201 Bell Hall, Buffalo, NY 14260, USA

(in-package :snip)


; =============================================================================
;
; <flagged node set> ::= { <flagged node> ... }
;
; -----------------------------------------------------------------------------
;
; PRIMITIVE      new.fns    :  --> <flagged node set>
;  ELEMENTS
;
; RECOGNIZERS    is.fns     : <universal> --> <boolean>
;                isnew.fns  : <flagged node set> --> <boolean>
;
; CONSTRUCTORS   insert.fns : <flagged node> x <flagged node set>
;                                 --> <flagged node set>
;                putin.fns  : <flagged node> x <flagged node set>
;                                 --> <flagged node set>
;                update.fns : <flagged node set> x <node> x <flag>
;                                 --> <flagged node set>
;
; SELECTORS      choose.fns : <flagged node set> --> <flagged node>
;                others.fns : <flagged node set> --> <flagged node set>
;                flag.fns   : <node> x <flagged node set> --> <flag>
;                support.fns: <node> x <flagged node set> --> <support>
;
; TESTS          ismemb.fns : <flagged node> x <flagged node set>
;                                 --> <boolean>          
;
; =============================================================================
;
; new.fns
; -------
;
;       arguments     : 
;
;       returns       : <flagged node set>
;
;       description   : returns a "new" <flagged node set>
;
;                                        written :  rgh  2/08/86
;                                        modified:
;
;
(defmacro new.fns ()
  `(new.Set))
;
;
; =============================================================================
;
; choose.fns
; ----------
;
;       arguments     : fns - <flagged node set>
;
;       returns       : <flagged node>
;
;       description   : returns an element of "fns"
;
;                                        written :  rgh  2/08/86
;                                        modified:
;
;
(defmacro choose.fns (fns)
  `(choose.Set ,fns))
;
;
;;; =============================================================================
;;;
;;; do.fns 
;;; ------
;;;
;;;       arguments     : var - a local variable
;;;                       fnsform - a form evaluating to a <flagged node set>
;;;                       resultform - a form evaluating to a <flagged node set>
;;;                       forms - a sequence of Lisp forms
;;;
;;;       returns       : <flagged node set> or nil
;;;
;;;       description   : Evaluates the sequence of forms in order once
;;;                       for val bound to sucessive elements of the value of fnsform
;;;                       Then returns the value of resultform.
;;;                       If resultform is omitted, returns nil.
;;;
;;;                                        written:  scs 03/03/88
;;;
(defmacro do.fns ((var fnsform &optional resultform) &body forms)
  `(do.set (,var ,fnsform ,resultform) ,@forms))
;;;
;;;
; =============================================================================
;
; is.fns
; ------
;
;       arguments     : u - <universal>
;
;       returns       : <boolean>
;
;       description   : returns "true" if "u" is a <flagged node set>,
;                       "false" otherwise
;
;                                        written :  rgh  2/08/86
;                                        modified:
;
;
(defmacro is.fns (u)
  `(and (is.Set ,u)
        (or (isnew.Set ,u)
            (is.fn (choose.Set ,u)))))
;
;
; =============================================================================
;
; isnew.fns
; ---------
;
;       arguments     : fns - <flagged node set>
;
;       returns       : <boolean>
;
;       description   : returns "true" if "fns" is a "new" <flagged node set>,
;                       "false" otherwise
;
;                                        written :  rgh  2/08/86
;                                        modified:
;
;
(defmacro isnew.fns (fns)
  `(isnew.Set ,fns))
;
;
; =============================================================================
;
; others.fns
; ----------
;
;       arguments     : fns - <flagged node set>
;
;       returns       : <flagged node set>
;
;       description   : returns all of "fns" except for the element that
;                       would be chosen by choose.fns
;
;                                        written :  rgh  2/08/86
;                                        modified:
;
;
(defmacro others.fns (fns)
  `(others.Set ,fns))
;
;
; =============================================================================
;
; ismemb.fns
; ----------
;
;       arguments     : fn  - <flagged node>
;                       fns - <flagged node set>
;
;       returns       : <boolean>
;
;       description   : returns "true" if "fn" is member of "fns", 
;                       "false" otherwise
;
;                                        written :  njm/cpf 10/18/88
;                                        modified:
;
;

(defmacro ismemb.fns (fn fns)
  `(do* ((fs ,fns (others.fns fs))
	 (f (choose.fns fs) (choose.fns fs))
	 (result nil))
	((or (isnew.fns fs) result) result)
     (if (and (iseq.n (node.fn ,fn) (node.fn f))
	      (isincluded.sup (support.fn ,fn) (support.fn f))
	      (eq (flag.fn ,fn) (flag.fn f)))
	 (setq result t))))

;
; =============================================================================
;
; insert.fns
; ----------
;
;       arguments     : fn - <flagged node>
;                       fns - <flagged node set>
;
;       returns       : <flagged node set>
;
;       description   : returns a <flagged node set> consisting of "fns",
;                       but with "fn" also included if it was not already
;
;                                        written :  rgh  2/08/86
;                                        modified:  njm/cpf 10/20/88
;
;
(defmacro insert.fns (fn fns)
  `(let ((result nil)
	 (finfns (new.fns)))
     (cond ((ismemb.fns ,fn ,fns) ,fns)
	   ((do.Set (f ,fns result)
	      (if (and (iseq.n (node.fn f) (node.fn ,fn))
		       (eq (flag.fn f) (flag.fn ,fn)))
		  (progn (setq finfns (putin.Set (merge.fn f ,fn)
						 finfns))
			 (setq result t))
		  (setq finfns (putin.Set f finfns))))
	    finfns)
	   (t (putin.Set ,fn ,fns)))))


;
; =============================================================================
;
; putin.fns
; ---------
;
;       arguments     : fn - <flagged node>
;                       fns - <flagged node set>
;
;       returns       : <flagged node set>
;
;       description   : returns a <flagged node set> consisting of "fns" with
;                       "fn" added to it.  No check is made to see if it was
;                       already there.
;
;                                        written :  rgh  2/13/86
;                                        modified:
;
;
(defmacro putin.fns (fn fns)
  `(putin.Set ,fn ,fns))
;
;
; =============================================================================
;
; update.fns
; ----------
;
;       arguments     : fns - <flagged node set>
;                       n - <node>
;                       sup - <support>
;                       flag - <flag>
;
;       returns       : <flagged node set>
;
;       description   : returns a <flagged node set> similar to "fns" but
;                       with the flag corresponding to "n" set to "flag";
;                       This function also updates the node a-support.
;
;                                        written :  rgh  4/03/86
;                                        modified:  cpf 10/07/88
;
;
(defun update.fns (fns n sup flag)
  (let ((fn (choose.fns fns)))
    (cond ((isnew.fns fns) fns)
	  ((iseq.n n (node.fn fn))
	   (putin.fns (make.fn n (merge.sup sup (support.fn fn)) flag)
		      (others.fns fns)))
	  (t
	   (putin.fns fn (update.fns (others.fns fns) n sup flag))))))
;
;
; =============================================================================
;
; flag.fns
; --------
;
;       arguments     : n - <node>
;                       fnset - <flagged node set>
;
;       returns       : <flag> | nil
;
;       description   : returns the <flag> associated with the <node> "n"
;                       in "fns" 
;
;       implementation: actually should only be used when it is known that
;                       "n" is a node in "fnset", so "nil" should neve be
;                       returned
;
;                                        written :  rgh  2/12/86
;                                        modified:
;
;
(defmacro flag.fns (n fnset)
  `(prog (fns fn)
	 (setq fns ,fnset)
      begin
	 (if (isnew.fns fns) (return))
	 (setq fn (choose.fns fns))
	 (if (iseq.n ,n (node.fn fn)) (return (flag.fn fn)))
	 (setq fns (others.fns fns))
	 (go begin)))
;
;
; =============================================================================
;
; support.fns
; -----------
;
;       arguments     : n - <node>
;                       fnset - <flagged node set>
;
;       returns       : <support> | nil
;
;       description   : returns the <support> associated with the <node> "n"
;                       in "fns" 
;
;       implementation: actually should only be used when it is known that
;                       "n" is a node in "fnset", so "nil" should neve be
;                       returned
;
;                                        written :  cpf 10/18/88
;                                        modified:
;
;
(defmacro support.fns (n fnset)
  `(prog (fset fn)
	 (setq fset ,fnset)
      begin
	 (if (isnew.fns fset) (return))
	 (setq fn (choose.fns fset))
	 (if (iseq.n ,n (node.fn fn)) (return (support.fn fn)))
	 (setq fset (others.fns fset))
	 (go begin)))
;
;
; =============================================================================



    
    




