;;; -*- Mode: Lisp; Syntax: Common-Lisp; Package: SNIP; Base: 10 -*-

;; Copyright (C) 1984--2004
;; Research Foundation of State University of New York

;; Version: $Id: report.lisp,v 1.7 2004/08/26 23:26:08 snwiz Exp $

;; This file is part of SNePS.

;; SNePS is free software; you may redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; SNePS is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with SNePS; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA, or to
;; Dr. Stuart C. Shapiro, Department of Computer Science and Engineering,
;; University at Buffalo, The State University of New York, 
;; 201 Bell Hall, Buffalo, NY 14260, USA

(in-package :snip)


; =============================================================================
;
; <report> ::= ( <substitution> <support> <sign> <signature> <node> <ct> )
;
; Where <ct> ::=  <context> | 'NIL
;
; A <report> is a report that a certain instance of a supported node N or of the
; negation of N is asserted in the network.
; <signature> is the node sending the report.
; When a report is sent into a channel, <node> is NIL.
; When a report emerges from a channel, and/or is placed in a *REPORTS* register,
;      If <sign> is POS, then <node> is the <substitution> instance of N;
;      If <sign> is NEG, then <node> is the <substitution> instance of the negation of N.
;
; -----------------------------------------------------------------------------
;
; RECOGNIZERS    is.rep              :  <universal> --> <boolean>
;                is-ant-to-rule.rep  :  <report> --> <boolean>
;                is-rule-to-cq.rep   :  <report> --> <boolean>
;                is-node-to-node.rep :  <report> --> <boolean>
;
; CONSTRUCTORS   make.rep    :  <substitution> x <support> x <sign> x <signature> 
;                                  x <node> x [<context>] --> <report>  
;
; SELECTORS      subst.rep     :  <report> --> <substitution>
;                support.rep   :  <report> --> <support>
;                sign.rep      :  <report> --> <sign>
;                signature.rep :  <report> --> <signature>
;                node.rep      :  <report> --> <node>
;                context.rep   :  <report> --> <context>
;
; UTILITY        addbinding.rep :  <mbind> x <report> --> <report>
;
; TESTS          iseq.rep  :  <report> x <report> --> <boolean>
;                ispos.rep :  <report> --> <boolean>
;                isneg.rep :  <report> --> <boolean>
;                hasct.rep :  <report> --> <boolean>
;
; =============================================================================
;


;
; =============================================================================
;
; make.rep
; --------
;
;       arguments     : subst - <substitution>
;                       sup - <support>
;                       sign - <sign>
;                       sig - <signature>
;                       n - <node>
;                       ct - <context> 
;
;       returns       : <report>
;
;       description   : returns a <report> consisting of the components
;                        passed as arguments
;
;                                        written :  rgh  7/27/85
;                                        modified:  rgh  3/09/86
;                                        modified:  cpf 10/07/88
;
(defmacro make.rep (subst sup sign sig n ct)
  `(list ,subst ,sup ,sign ,sig ,n ,ct))
; 
;
; =============================================================================
;
; subst.rep
; ---------
;
;       arguments     : report  -  <report>
;
;       returns       : <substitution>
;
;       description   : selects the <substitution> of the report
;
;                                        written :  rgh 05/09/85
;                                        modified:  rgh 06/11/85
;
;
(defmacro subst.rep (report)
   `(first ,report))
;
;
; =============================================================================
;
; support.rep
; -----------
;
;       arguments     : report  -  <report>
;
;       returns       : <support>
;
;       description   : selects the <support> of the report
;
;                                        written :  cpf 10/07/88
;                                        modified:  
;
;
(defmacro support.rep (report)
   `(second ,report))
;
;
; =============================================================================
;
; sign.rep
; --------
;
;       arguments     : report - <report>
;
;       returns       : <sign>
;
;       description   : selects the <sign> of the report
;
;                                        written :  rgh 07/24/85
;                                        modified:  cpf 10/07/88
;
;
(defmacro sign.rep (report)
   `(third ,report))
;
;
; =============================================================================
;
; signature.rep
; -------------
;
;       arguments     : report  -  <report>
;
;       returns       : <signature>  ( = <node> )
;
;       description   : selects the <signature> of the report
;
;                                        written :  rgh 05/09/85
;                                        modified:  rgh 06/11/85
;                                        modified:  cpf 10/07/88
;
(defmacro signature.rep (report)
   `(fourth ,report))
;
;
; =============================================================================
;
; node.rep
; --------
;
;       arguments     : report  -  <report>
;
;       returns       : <node>
;
;       description   : selects the <node> of the report
;
;                                        written :  rgh  3/09/86
;                                        modified:  cpf 10/07/88
;
;
(defmacro node.rep (report)
   `(fifth ,report))
;
;
; =============================================================================
;
; context.rep
; -----------
;
;       arguments     : report  -  <report>
;
;       returns       : <context>
;
;       description   : selects the <node> of the report
;
;                                        written :  rgh  3/09/86
;                                        modified:  cpf 10/07/88
;
;
(defmacro context.rep (report)
   `(sixth ,report))
;
; =============================================================================
;
; is.rep
; ------
;
;       arguments     : u - <universal>
;
;       returns       : <boolean>
;
;       description   : returns "true" is "u" is a <report>,
;                               "false" otherwise
;
;                                        written :  rgh  7/27/85
;                                        modified:  rgh  3/09/86
;                                        modified:  cpf 10/07/88
;
(defmacro is.rep (u)
  `(and (listp ,u)
        (is.sbst (first ,u))
	(is.sup (second ,u))
        (is.sign (third ,u))
        (is.sig (fourth ,u))
        ((or (null (nth 4 ,u)) (is.n (nth 4 ,u))))))
;
;

; =============================================================================
;
; is-ant-to-rule.rep
; ------------------
;
;       arguments     : report  -  <report>
;
;       returns       : <boolean>
;
;       description   : determines whether or not the report is from a node
;                       in antecedent position to a dominating rule node
;
;       implementation: checks to see if the report was sent by a node which
;                       is at the end of a ant, arg, or &ant arc, since these
;                       are the only arcs which point to the antecedent of a
;                       rule
;
;                                        written :  rgh  6/11/85
;                                        modified:  rgh  4/20/86
;                                                    hi  3/31/99
;
(defmacro is-ant-to-rule.rep (report)
  `(let ((sender (signature.rep ,report)))
      (and (is.n sender)
           (or (ismemb.ns sender (nodeset.n *NODE* 'ant))
               (ismemb.ns sender (nodeset.n *NODE* 'arg))
               (ismemb.ns sender (nodeset.n *NODE* 'when))
               (ismemb.ns sender (nodeset.n *NODE* 'whenever))
               (ismemb.ns sender (nodeset.n *NODE* '&ant))))))
;
;; =============================================================================
;
; is-cq-to-rule.rep
; -----------------
;
;       arguments     : report  -  <report>
;
;       returns       : <boolean>
;
;       description   : determines whether or not the report is from a node
;                       in consequent position to a dominating rule node
;
;       implementation: checks to see if the report was sent by a node which
;                       is at the end of a cq, arg, or dcq arc, since these
;                       are the only arcs which point to the antecedent of a
;                       rule
;
;                                        written :  njm/cpf 11/04/88
;                                        modified:  
;
;
(defmacro is-cq-to-rule.rep (report)
  `(let ((sender (signature.rep ,report)))
      (and (is.n sender)
           (or (ismemb.ns sender (nodeset.n *NODE* 'cq))
               (ismemb.ns sender (nodeset.n *NODE* 'arg))
               (ismemb.ns sender (nodeset.n *NODE* 'dcq))))))
;
;
; =============================================================================
;
; is-rule-to-cq.rep
; -----------------
;
;       arguments     : report  -  <report>
;
;       returns       : <boolean>
;
;       description   : determines whether or not the report is from a rule
;                       node to a node in consequent position of the rule
;
;       implementation: checks to see if the report was sent by a node which
;                       is at the end of a cq-, arg-, or dcq- arc, since
;                       these are the only arcs which point to a rule for
;                       which the current node is in consequent position
;
;                                        written :  rgh  6/11/85
;                                        modified:  rgh  4/20/86
;
;
(defmacro is-rule-to-cq.rep (report)
  `(let ((sender (signature.rep ,report)))
      (and (is.n sender)
           (or (ismemb.ns sender (nodeset.n *NODE* 'cq-))
               (ismemb.ns sender (nodeset.n *NODE* 'arg-))
               (ismemb.ns sender (nodeset.n *NODE* 'dcq-))))))
;
;
; =============================================================================
;
; is-node-to-node.rep
; -------------------
;
;       arguments     : report  -  <report>
;
;       returns       : <boolean>
;
;       description   : determines whether or not the report is from a node
;                       at the other end of a match channel
;
;       implementation: checks to see if the report is neither of the type
;                       ant-to-rule nor rule-to-cq, since they are the only
;                       other possible types of reports
;
;                                        written :  rgh  6/11/85
;                                        modified:  rgh  4/20/86
;
;
(defmacro is-node-to-node.rep (report)
  `(and (is.n (signature.rep ,report))
        (not (is-ant-to-rule.rep ,report))
        (not (is-rule-to-cq.rep ,report))))
;
;
;
; =============================================================================
;
; addbinding.rep
; --------------
;
;       arguments     : mb - <mbind>
;                       rep - <report>
;
;       returns       : <report>
;
;       description   : adds the binding "mb" to the <substitution> of "rep"
;
;                                        written :  rgh  8/05/85
;                                        modified:  rgh  3/09/86
;
;
(defmacro addbinding.rep (mb rep)
  `(make.rep (putin.sbst ,mb (subst.rep ,rep))
	     (support.rep ,rep)
             (sign.rep ,rep)
             (signature.rep ,rep)
             (node.rep ,rep)
	     (context.rep ,rep)))
;
;
; =============================================================================
;
; iseq.rep
; --------
;
;       arguments     : r1 - <report>
;                       r2 - <report>
;
;       returns       : <boolean>
;
;       description   : returns "true" if "r1" and "r2" are equal
;
;                                        written :  rgh 11/30/85
;                                        modified:  rgh  3/09/86
;                                        modified:  scs  3/29/88
;                                        modified:  cpf 10/07/88
;
(defun iseq.rep (r1 r2)
  (and (iseq.sbst (subst.rep r1) (subst.rep r2))
       (iseq.sup (support.rep r1) (support.rep r2))
       (iseq.sign (sign.rep r1) (sign.rep r2))
       (iseq.n (signature.rep r1) (signature.rep r2))
       (sneps:iseq.ct (context.rep r1) (context.rep r2))
       (let ((n1 (node.rep r1)) (n2 (node.rep r2)))
	 (or (and (null n1) (null n2))
	     (and (not (null n1))
		  (not (null n2))
		  (iseq.n n1 n2))))))
;
;
; =============================================================================
;
; ispos.rep
; ---------
;
;       arguments     : report - <report>
;
;       returns       : <boolean>
;
;       description   : returns "true" if "report" is a positive <report>,
;                       "false" otherwise
;
;                                        written :  rgh  2/12/86
;                                        modified:
;
;
(defmacro ispos.rep (report)
  `(eq (sign.rep ,report) 'POS))
;
;
; =============================================================================
;
; isneg.rep
; ---------
;
;       arguments     : report - <report>
;
;       returns       : <boolean>
;
;       description   : returns "true" if "report" is a negative <report>,
;                       "false" otherwise
;
;                                        written :  rgh  2/12/86
;                                        modified:
;
;
(defmacro isneg.rep (report)
  `(eq (sign.rep ,report) 'NEG))
;
;
; =============================================================================
;
; hasct.rep
; ---------
;
;       arguments     : report - <report>
;
;       returns       : <boolean>
;
;       description   : returns "true" if "report" has context information
;                       other than nil, "false" otherwise
;
;                                        written :  njm/cpf 10/26/88
;                                        modified:
;
;
(defmacro hasct.rep (report)
  `(sneps:is.ct (context.rep ,report)))
;
;
; =============================================================================



    
    




