;;; -*- Mode: Lisp; Syntax: Common-Lisp; Package: SNIP; Base: 10 -*-

;; Copyright (C) 1984--2004
;; Research Foundation of State University of New York

;; Version: $Id: request.lisp,v 1.7 2004/08/26 23:26:08 snwiz Exp $

;; This file is part of SNePS.

;; SNePS is free software; you may redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; SNePS is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with SNePS; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA, or to
;; Dr. Stuart C. Shapiro, Department of Computer Science and Engineering,
;; University at Buffalo, The State University of New York, 
;; 201 Bell Hall, Buffalo, NY 14260, USA

(in-package :snip)


; =============================================================================
;
; <request>  ::=  <channel>
;
; =============================================================================
;
; RECOGNIZERS    is-cq-to-rule.req   : <request> --> <boolean>
;                is-rule-to-ant.req  : <request> --> <boolean>
;                is-node-to-node.req : <request> --> <boolean>
;                is-from-user.req    : <request> --> <boolean>
;
; =============================================================================
;
; is-cq-to-rule.req
; -----------------
;
;       arguments     : request  -  <request>
;
;       returns       : <boolean>
;
;       description   : determines whether or not the request is from a node
;                       in consequent position to a dominating rule node
;
;       implementation: checks to see if the request was sent by a node which
;                       is at the end of a cq, arg, or dcq arc, since these
;                       are the only arcs which point to the consequent of a
;                       rule.
;
;                                        written :  rgh  6/11/85
;                                        modified:  rgh  3/08/86
;                                                     dk 6/2/93
; added the check to an IF arc for a DO-IF rule -: dk
;
;
(defmacro is-cq-to-rule.req (request)
   `(let ((sender (destination.ch ,request)))
       (and (is.n sender)
            (or (ismemb.ns sender (nodeset.n *NODE* 'cq))
                (ismemb.ns sender (nodeset.n *NODE* 'arg))
                (ismemb.ns sender (nodeset.n *NODE* 'if)) ; a DO-IF rule
                (ismemb.ns sender (nodeset.n *NODE* 'dcq))))))
;
; 
; =============================================================================
;
; is-rule-to-cq.req
; ------------------
;
;       arguments     : request  -  <request>
;
;       returns       : <boolean>
;
;       description   : determines whether or not the request is from a rule
;                       node to a node in consequent position of the rule
;
;       implementation: checks to see if the request was sent by a node which
;                       is at the end of a cq-, or arg- arc, since
;                       these are the only arcs which point to a rule for
;                       which the current node is in consequent position.
;
;                                        written :  cpf     11/04/88
;                                        modified:  cpf/njm 07/12/89
;
;
(defmacro is-rule-to-cq.req (request)
  `(let ((sender (destination.ch ,request)))
    (and (is.n sender)
	 (or (ismemb.ns sender (nodeset.n *NODE* 'sneps::cq-))
	     (ismemb.ns sender (nodeset.n *NODE* 'sneps::arg-))))))
;
;
; =============================================================================
;
; is-rule-to-ant.req
; ------------------
;
;       arguments     : request  -  <request>
;
;       returns       : <boolean>
;
;       description   : determines whether or not the request is from a rule
;                       node to a node in antecedent position of the rule
;
;       implementation: checks to see if the request was sent by a node which
;                       is at the end of a ant-, arg-, or &ant- arc, since
;                       these are the only arcs which point to a rule for
;                       which the current node is in antecedent position
;
;                                        written :  rgh  6/11/85
;                                        modified:  rgh  3/08/86
;
;
(defmacro is-rule-to-ant.req (request)
   `(let ((sender (destination.ch ,request)))
       (and (is.n sender)
            (or (ismemb.ns sender (nodeset.n *NODE* 'ant-))
                (ismemb.ns sender (nodeset.n *NODE* 'arg-))
                (ismemb.ns sender (nodeset.n *NODE* '&ant-))))))
;
;
; =============================================================================
;
; is-node-to-node.req
; -------------------
;
;       arguments     : request  -  <request>
;
;       returns       : <boolean>
;
;       description   : determines whether or not the request is from a node
;                       at the other end of a match channel
;
;       implementation: checks to see if the request is neither of the type
;                       cq-to-rule nor rule-to-ant, since they are the only
;                       other possible types of requests
;
;                                        written :  rgh  6/11/85
;                                        modified:
;
;
(defmacro is-node-to-node.req (request)
  `(and (not (is-cq-to-rule.req ,request))
	(not (is-rule-to-ant.req ,request))
	(not (is-rule-to-cq.req ,request))))
;
;
; =============================================================================
;
; is-from-user.req
; ----------------
;
;       arguments     : request - <request>
;
;       returns       : <boolean>
;
;       description   : returns "true" if "request" is from the USER-PROCESS
;                       which was set up by a call to deduce, "false"
;                       otherwise
;
;                                        written :  rgh  3/08/86
;                                        modified:
;
;
(defmacro is-from-user.req (request)
  `(is-user.dest (destination.ch ,request)))
;
;
; =============================================================================



    
    




