;;; -*- Mode: Lisp; Syntax: Common-Lisp; Package: SNIP; Base: 10 -*-

;; Copyright (C) 1993--2004
;; Research Foundation of State University of New York

;; Version: $Id: mental-acts.lisp,v 1.7 2004/08/26 23:26:12 snwiz Exp $

;; This file is part of SNePS.

;; SNePS is free software; you may redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; SNePS is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with SNePS; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA, or to
;; Dr. Stuart C. Shapiro, Department of Computer Science and Engineering,
;; University at Buffalo, The State University of New York, 
;; 201 Bell Hall, Buffalo, NY 14260, USA

(in-package :snip)


;;;
;;; Representation
;;;
;;;   believe-act = action believe
;;;         	    object1 prop
;;;   believe-act = action believe
;;;         	    object1 prop


;;; HC: Believing of the effects of an act should really be done simultaneously
;;; (or in parallel) instead of the current method that uses a DO-ALL in
;;; the function `schedule-believe-effects'.  The reason is that an act can
;;; have multiple effects, some of which might also be represented by
;;; rules.  For example, the rule Held(BLOCK) => ~Clear(BLOCK) in conjunction
;;; with the effects Held(BLOCK) and ~Clear(BLOCK) of the pickup action will
;;; lead to a contradiction if after picking up a block we first believe
;;; Held(BLOCK) and forward-chain on it, since that will derive ~Clear(BLOCK)
;;; which will contradict the still not disbelieved Clear(BLOCK).

;;; In preparation for changing this the `believe' function now uses
;;; `believe-propositions' which believes a set of propositions simultaneously.

;;; Changed FLJ/SCS 8/1/04 to do the following...
;;; Provides an error message, instead of causing a Lisp error, if no proposition.
(defun believe (believe-act)
  "Causes the propositions in object1(BELIEVE-ACT) to be believed."
  (if (isnew.ns (nodeset.n believe-act 'object1))
      (sneps:sneps-error "Attempt to perform believe with no proposition."
			 'snip 'believe)
    (believe-propositions (nodeset.n believe-act 'object1))))



;;; Want `believe P' always to do: retract ~P; assert P.
;;; Modified by flj on 2/12/04 for scs

(defun believe-propositions (propositions)
  "Causes PROPOSITIONS to be believed quasi-simultaneously.
   By `simultaneously' is meant that before believing any PROPOSITIONS
   all currently believed negations to any of them will be disbelieved,
   as well as all currently believed linked propositions."
  (let* ((sneps:*with-snepsul-eval-function*
	  ;; Make sure no tracing occurs in #! contexts:
          #'sneps:with-snepsul-standard-eval))
    ;; Disbelieve negations:
    (do.ns (proposition propositions)
	   (disbelieve-negation proposition)
	   (disbelieve-linked proposition))
    ;; Assert propositions:
    (do.ns (proposition propositions)
	   (cond ((isassert.n proposition)
		  (plantrace
		   "I already believe " (makeone.ns proposition) (new.restr)))
		 (t #!((! ~proposition :context ~crntct))
		    (plantrace "Believe " (makeone.ns proposition) (new.restr)))))
    ;; Forward-chain on newly believed propositions:
    (dynamic-add* propositions)))

;;;  Added by flj on 2/12/04 for scs
(defun disbelieve-linked (proposition)
  "Retracts any believed propositions linked to PROPOSITION."
  (let ((linked-propositions
	 #!((- (find arg- 
		     (findassert max 1 arg ~proposition :context ~crntct))
	       ~proposition))))
    ;; Retract linked propositions
    (do.ns (linked-proposition linked-propositions)
	   (when (isassert.n linked-proposition)
	     (disbelieve-propositions (makeone.ns linked-proposition))))))


;;; Changed scs/flj 6/29/04 to use sneps:negate.n
;;; Changed scs/flj 8/01/04 for following reason...
;;; Now that sneps:contradictory-nodes.n uses find-negate instead of negate.n,
;;;    it might return an empty nodeset.
(defun disbelieve-negation (proposition
			    &aux
			    (not-prop (sneps:contradictory-nodes.n proposition)))
  "Retracts believed propositions that are contradictions of PROPOSITION."
  (unless (isnew.ns not-prop)
    (disbelieve-propositions 
     (sneps:remove-if-not.ns #'isassert.n not-prop))))

(defun disbelieve (disbelieve-act)
  "Causes the propositions in object1(DISBELIEVE-ACT) to be disbelieved."
  (disbelieve-propositions (nodeset.n disbelieve-act 'object1)))

;;;  Want `disbelieve P' always to do just retract P.
;;;  Modified by flj on 2/12/04 for scs

(defun disbelieve-propositions (propositions)
  "Causes PROPOSITIONS to be disbelieved."
  (let ((sneps:*with-snepsul-eval-function*
         #'sneps:with-snepsul-standard-eval))
    (do.ns (proposition propositions)
	   (cond ((isassert.n proposition)
		  (cond ((ismemb.ns proposition (sneps:context-hyps crntct))
			 (plantrace "Disbelieve "
				    (makeone.ns proposition) (new.restr))
			 #3!((remove-from-context ~proposition ~crntct)))
			(t (plantrace
			    "Can't disbelieve derived proposition "
			    (makeone.ns proposition) (new.restr)))))
		 (t (plantrace
		     "already didn't believe "
		     (makeone.ns proposition)
		     (new.restr)))))))

;;; For backward compatibility:

(defun forget (forget-act)
  "Alias for `disbelieve'."
  (disbelieve forget-act))



    
    




