;;; -*- Mode: Lisp; Syntax: Common-Lisp; Package: SNIP; Base: 10 -*-

;; Copyright (C) 1984--2004
;; Research Foundation of State University of New York

;; Version: $Id: nrn-requests.lisp,v 1.8 2004/08/26 23:26:13 snwiz Exp $

;; This file is part of SNePS.

;; SNePS is free software; you may redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; SNePS is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with SNePS; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA, or to
;; Dr. Stuart C. Shapiro, Department of Computer Science and Engineering,
;; University at Buffalo, The State University of New York, 
;; 201 Bell Hall, Buffalo, NY 14260, USA

(in-package :snip)


; =============================================================================
;
; process-requests.non-rule
; -------------------------
;
;       nonlocal-vars : various registers of the current *NODE*
;                         *NODE*, *REQUESTS*
;
;       description   : for each incoming request, the requested channel is
;                       inserted into the *OUTGOING-CHANNELS* register, and
;                       then:
;                          - if the current node is asserted, a report is
;                             sent through the channel
;                          - all known instances are sent through the channel
;                          - if this request is not already being worked on,
;                             requests are passed back to potential sources
;                             of answers
;
;       side-effects  : registers affected:  *REQUESTS*, *OUTGOING-CHANNELS*
;
;                                        written :  rgh 10/06/85
;                                        modified:  rgh 11/18/85
;                                                   rgh  2/02/86
;                                                   rgh  2/09/86 
;                                                   rgh  4/26/86
;                                                   scs  4/20/88
;                                                   njm 10/23/88
;                                                   cpf/njm 10/24/88
;                                                   ssc  5/10/89
;
; 
(defun process-requests.non-rule ()
  (let ((requests *REQUESTS*)
	(remark-sent nil)
	 pathfrom-result)
    (declare (special remark-sent pathfrom-result))
    (setq *REQUESTS* (new.chset))
    (do.set (ch requests t)
      (process-one-request.non-rule ch))))

;
; 
; =============================================================================
;
; process-one-request.non-rule
; ----------------------------
;
;       arguments     : ch - <channel>
;
;       nonlocal-vars : *NODE*
;
;       description   : processes the single request "ch"
;
;                                        written :  rgh  2/09/86
;                                        modified:  rgh  3/09/86
;                                                   rgh  3/27/86
;                                                   rgh  4/26/86
;                                                   scs  6/23/88
;                                                   njm/cpf 10/21/88
;
(defun process-one-request.non-rule (ch)
  (let ((restr nil)
	(any-instances-sent nil)
	(crntct (context.ch ch)))
    (declare (special any-instances-sent crntct remark-sent))
    (install-channel ch)
    (cond ((isassert.n *NODE*)
	   (unless remark-sent     
	     (remark '"~%I know" (makeone.ns *NODE*) (new.restr))
	     (setq remark-sent t))
	   (send-reports (makeone.repset (make.rep
					   (new.sbst)
					   (filter.sup (sneps:node-asupport *NODE*)
						       crntct)
					   'POS
					   *NODE*
					   nil
					   crntct))
			 ch))
	  (t (send-known-instances ch)
	     (when (or (not any-instances-sent) (is-wh-question (filter.ch ch)))
	       (setq restr (ch-to-restr ch))
	       (when (not-working-on restr crntct ch)
		 (send-rule-use-requests restr crntct (destination.ch ch))
		 (when (and (or (is-rule-to-ant.req ch)
				(is-rule-to-cq.req ch)
				(is-from-user.req ch))
			    (enough-resources))
		   (decrease-resources)
		   (remark '"~%I wonder if"
			   (makeone.ns *NODE*)
			   restr
			   (context.ch ch))
		   (send-node-to-node-requests restr crntct)))
	       (when (and (is-case-of-introduction ch)
			  (not-working-on-introdution restr crntct))
		 (initiate-introduction.rule ch)))))))
;
;
; =============================================================================
;
; is-wh-question
; --------------
;
;       arguments     : substitution - <substitution>
;
;       returns       : <boolean>
;
;       description   : returns "true" if any of the variables of
;                       "substitution" are bound to variables, thus indicating
;                       that the request is for as many answers as can be
;                       found, and "false" if each variable is bound to a
;                       constant, indicating a yes/no request.  Also checks
;                       all free variables dominated by the current node
;                       to see whether or not they are all bound to constants.
;
;                                        written :  rgh  3/27/86
;                                        modified:  rgh  4/18/86
;
;
(defun is-wh-question (substitution)
  (do* ((ns (freevars.n *NODE*)
	    (others.ns ns))
	(bndg (term.sbst (choose.ns ns) substitution)
	      (term.sbst (choose.ns ns) substitution)))
       ((or (isnew.ns ns)
	    (null bndg)
	    (not (is.n bndg))
	    (isvar.n bndg)
	    (ispat.n bndg))
	(not (isnew.ns ns)))))
;
;
; =============================================================================
;
; send-request
; ------------
;
;       arguments     : req - <request>
;                       n - <node>
;                       restr - <restriction>
;
;       nonlocal-vars : the *REQUESTS* register of the activation process of
;                       the node "n", and the *INCOMING-CHANNELS* register
;                       of the current node process
;
;       description   : actually sends the (request) message "req" to the
;                       node "n".  Also installs the incoming channel
;                       information.
;
;       side-effects  : inserts "req" into the *REQUESTS* register of "n"
;                       and updates *INCOMING-CHANNELS*
;
;                                        written :  rgh 10/07/85
;                                        modified:  rgh 11/24/85
;                                                   rgh  4/13/86
;                                                   njm/cpf 10/21/88
;                                                   choi 9/19/90
;
(defun send-request (req n restr)
  (let (pr)
    (activate.n n)
    (setq pr (activation.n n))
    (when (eq (regfetch pr '*NAME*) 'NUM-QUANT.RULE)
      (let ((new-sbst 
	      (compl.Set (subst.restr restr)
			 (restrict.sbst (subst.restr restr)
					(quantified-vars.n (regfetch pr '*NODE*))))))
	
	(setq restr (make.restr new-sbst))
	(setq req (make.ch new-sbst
			   (switch.ch req)
			   (context.ch req)
			   (destination.ch req)
			   (valve.ch req)))))
    (regstore pr
	      '*REQUESTS*
	      (insert.chset req (regfetch pr '*REQUESTS*)))
    (regstore pr '*PRIORITY* 'LOW)
    (setq *INCOMING-CHANNELS*
	  (insert.feedset (make.feeder restr
				       (context.ch req)
				       n
				       (valve.ch req))
			  *INCOMING-CHANNELS*))
    (initiate pr)))
;
;
; =============================================================================
;
; send-known-instances
; --------------------
;
;       arguments     : ch - <channel>
;
;       returns       : <boolean>
;
;       nonlocal-vars : the *KNOWN-INSTANCES* and *NODE* registers of the
;                       current node, and "any-instances-sent"
;
;       description   : sends a report of each of the known instances of the
;                       current node through the channel "ch"
;
;
;                                        written :  rgh 10/07/85
;                                        modified:  rgh  3/09/86
;                                                   rgh  3/27/86
;                                                   scs  3/24/88
;                                                   scs  4/20/88
;                                                   njm/cpf 10/21/88
;                                                   njm 10/11/88
;
(defun send-known-instances (ch)
  (declare (special any-instances-sent))
  (do.set (inst *KNOWN-INSTANCES* any-instances-sent)
    (when (try-to-send-report (make.rep (subst.inst inst)
					(support.inst inst)
					(sign.inst inst)
					*NODE*
					nil
					(context.ch ch))
			      ch)
      (setq any-instances-sent t))))
;
;
; =============================================================================
;
; not-working-on
; --------------
;
;       arguments     : restr - <restriction>
;                       ct    - <context>
;                       ch    - <channel>
;
;       returns       : <boolean>
;
;       nonlocal-vars : the *INCOMING-CHANNELS* register of the current node
;
;       description   : returns "true" if there is NOT an incoming channel to the
;                       current node which will supply instances satisfying
;                       the restriction "restr" in the context "ct"; "false"
;                       otherwise
;
;                                        written :  rgh 10/07/85
;                                        modified:  scs 03/24/88
;                                                   njm 10/23/88
;                                                   cpf/njm 07/12/89
;
;
;
(defun not-working-on (restr ct ch)
  (do.set (feeder *INCOMING-CHANNELS* t)
    (when (and (equivalent.restr restr (restriction.feeder feeder))
	       (issubset.ct ct (context.feeder feeder))
	       ;;
	       ;; If the request does not came by Match-channel, it verifies
	       ;; first if it performed a match in the past before saying
	       ;; it is working on it.
	       ;;
	       (if (or (is-rule-to-ant.req ch)
		       (is-rule-to-cq.req ch)
		       (is-from-user.req ch))
		   (let ((source (source.feeder feeder)))
		     (not (or (ismemb.ns source (nodeset.n *NODE* 'sneps:cq-))
			      (ismemb.ns source (nodeset.n *NODE* 'sneps:arg-)))))
		   t))
      (return nil))))
;
;
; =============================================================================
;
; send-rule-use-requests
; ----------------------
;
;       arguments     : restriction - <restriction>
;                       context - <context>
;                       prev-requester - <node> or 'USER
;
;       returns       : <boolean>
;
;       nonlocal-vars : the *NODE* register of the current node
;
;       description   : sends requests for the instance(s) requested by the
;                       <restriction> given as the argument to the function
;                       to all rule nodes for which the current node is in
;                       consequent position except for prev-requester. The
;                       requests context is the one specified by "context".
;
;                                        written :  rgh 10/07/85
;                                        modified:  rgh 11/18/85
;                                                   rgh  4/13/86
;                                                   scs  4/22/88
;                                                   scs  6/23/88
;                                                   njm/cpf 10/21/88
;                                                   njm 10/23/88
;                                                   ssc  5/10/89
;                                                    dk  6/2/93
; (added the IF- arcs for the DOIF transformer rules -: dk)
;
;
;
(defun send-rule-use-requests (restriction context prev-requester)
  (let ((outgoing-request (make.ch (subst.restr restriction)
				   (new.sbst)
				   context
				   *NODE*
				   'OPEN)))
    (declare (special pathfrom-result))
    (broadcast-request
      outgoing-request
      (if (eq prev-requester 'USER)
	  (or pathfrom-result
	      (setq pathfrom-result
		    (sneps::pathfrom '(sneps::or cq- arg- dcq- if-)
				     *NODE*)))
	  (remove.ns prev-requester
		     (or pathfrom-result
			 (setq pathfrom-result
			       (sneps::pathfrom '(sneps::or cq- arg- dcq- if-)
						*NODE*)))))
      restriction)))
;
;
; =============================================================================
;
; broadcast-request
; -----------------
;
;       arguments     : req - <request>
;                       ns - <node set>
;                       restr - <restriction>
;
;       returns       : <boolean>
;
;       description   : sends "req" to each of the nodes in "ns"
;                       ("restr" is the desired restriction)
;
;                                        written :  rgh 10/07/85
;                                        modified:  rgh  4/13/86
;                                        modified:  scs  3/24/88
;
;
(defun broadcast-request (req ns restr)
  (do.ns (nde ns t)
    (send-request req nde restr)))
;
;
; =============================================================================
;
; send-node-to-node-requests
; --------------------------
;
;       arguments     : restriction - <restriction>
;                       context     - <context>
;
;       returns       : <boolean>
;
;       nonlocal-vars : the *NODE* register of the current node
;
;       description   : calls match and sends requests for the instance(s)
;                       requested by "restriction" to all nodes which match
;                       the current node
;
;                                        written :  rgh 10/07/85
;                                        modified:  rgh 11/18/85
;                                                   rgh 11/24/85
;                                                   rgh  4/13/86
;                                                   scs  3/24/88
;                                                   njm/cpf 10/21/88
;
(defun send-node-to-node-requests (restriction context)
  ;;"Modified to facilitate knowledge shadowing by using the most general 
  ;; common instances (mgci) of two patterns.  It now sends requests only 
  ;; to those nodes that are determined not to be redundant by checking  
  ;; the mgci of the current node (source node) and the matched node 
  ;; (target node).       modified by choi: 2/25/92"
  (let* ((rsbst (subst.restr restriction))
	 (rnode (match::applysubst *NODE* rsbst)))
    ;; if the requesting node 'rnode' (*NODE* restricted by 'rsbst')
    ;; is already asserted,
    ;; send request directly to 'rnode'  without doing match.
    (if (isassert.n rnode)
	(send-request (make.ch nil rsbst context *NODE* 'OPEN) 
		      rnode restriction)
      ;; otherwise, call match and check if the mgci of the current node
      ;; (restricted by 'rsbst') and the target node is asserted.
      ;; send request to such a node that mgci is not asserted.
      (do.supmatchingset (sup (match-in-context *NODE* context rsbst) t)
        (let ((tnode (tnode.supmatching sup))
	      (tsbst (target-sub.supmatching sup))
	      (ssbst (source-sub.supmatching sup)))
	  (unless (or (eq tnode *NODE*)
		      ;; check if the mgci of the current and target node
		      ;; is asserted so that it can block the request
		      (and (ispat.n tnode)
			   (is-mgci-asserted tnode tsbst)))
	    (send-request (make.ch tsbst ssbst context *NODE* 'OPEN)
			  tnode restriction)))))))


;;; Change to call sneps:negate.n instead of snebr::negator-of
(defun is-mgci-asserted (tnode tsbst)
  ;;"The most general common instance (mgci) of two patterns S and T 
  ;; is Ss or Tt, where s is a source binding and t is a target binding.
  ;; Obtaining mgci in SNIP is implemented by applying the target binding 
  ;; to the target node.
  ;; This procedure will return T if the mgci or the negation of the mgci
  ;; is asserted in the current context."
  (let ((mgci (match::applysubst tnode tsbst)))
    (and (not (ispat.n mgci))
	 (or (isassert.n mgci)
	     (isassert.n (sneps:negate.n mgci))))))

;
;
; =============================================================================



    
    




