;;; -*- Mode: Lisp; Syntax: Common-Lisp; Package: XGINSENG; Base: 10 -*-

;; Copyright (C) 1984--2004
;; Research Foundation of State University of New York

;; Version: $Id: input.lisp,v 1.8 2004/08/26 23:26:21 snwiz Exp $

;; This file is part of SNePS.

;; SNePS is free software; you may redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.

;; SNePS is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.

;; You should have received a copy of the GNU General Public License
;; along with SNePS; see the file COPYING.  If not, write to
;; the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA, or to
;; Dr. Stuart C. Shapiro, Department of Computer Science and Engineering,
;; University at Buffalo, The State University of New York, 
;; 201 Bell Hall, Buffalo, NY 14260, USA

(in-package :xginseng)


;;;;  Contains definitions for functions and objects that permit input
;;;;  to snepsul throught the garnet package


; ==================================================================================
;
; CREATE_BLANK_NODE
; -----------------
;
;          Arguments         :   <two-point-interactor>, <points-list>
;
;          Returns           :   A garnet node unassociated with any SNePS node
;
;          Description       :   This function is given a points list of the form
;                                (x1 y1 x2 y2) and returns a blank node with those
;                                coordinates
;
;          Dependencies      :   Called by the two-point-interactor
;                                create_blank_node_inter (defined in xginseng.lisp)
;
; ==================================================================================

(defun create_blank_node (inter points-list)
  (declare (ignore inter))
  (when points-list
	(let ((obj (intern (symbol-name (gensym "NODE")) 'xginseng)))
	  (create-instance obj node
			   (:box (copy-list points-list))
			   (:name "  ")
			   (:down-cables nil)
			   (:up-cables nil)
			   (:sneps-node nil)
			   (:sneps-command nil))
	  (push (eval obj) (g-value display-window :nodes-on-display))
	  (push (eval obj) (g-value display-window 
				    :temporary-nodes-and-arcs-on-display))
	  (opal:add-component (g-value display-window :inner-aggregate) 
			      (eval obj))
	  (eval obj))))




; ==================================================================================
;
; CREATE_BLANK_ARC
; -----------------
;
;          Arguments         :   <two-point-interactor>, <points-list>
;
;          Returns           :   A garnet-arc anchored at two existing garnet
;                                nodes. 
;
;          Description       :   This function is given a points list of the form
;                                (x1 y1 x2 y2) and creates a new arc that becomes
;                                attached to the two nodes that exist at those
;                                coordinates
;
;          Dependencies      :   Called by the two-point-interactor
;                                create_blank_arc_inter (defined in xginseng.lisp)
;
; ==================================================================================


(defun create_blank_arc (inter points-list)
  (declare (ignore inter))
  (when points-list
					;>>> First see if the two points are above
					;>>> two existing garnet nodes.  If not,
					;>>> beep an error message and exit
	(let ((obj (intern (symbol-name (gensym "ARC")) 'xginseng))
	      ;;get the identity of the from node
	      (from-node (opal:point-to-component 
			  (g-value display-window :inner-aggregate)
			  (first points-list)
			  (second points-list) :type node))
	      ;;get the identity of the to node
	      (to-node (opal:point-to-component 
			(g-value display-window :inner-aggregate)
			(third points-list)
			(fourth points-list) :type node)))
	  (if (or (null from-node)
		  (null to-node)
		  (eq from-node to-node))
	      (inter:beep)
	    (block false-clause-make-arc
		   (create-instance obj arc
					;>>> Set pointer fields in the arc to
					;>>> the source and destination nodes
				    (:from-node from-node)
				    (:to-node to-node)
					;>>> Calculate geometry of node-arc
					;>>> configuration (written by 
					;>>> Martin Zaidel)
				    (:theta 
				     (o-formula 
				      (atan (- (gvl :to-node :left) 
					       (gvl :from-node :left))
					    (- (gvl :to-node :top) 
					       (gvl :from-node :top)))))
					;distance between center of 
			                ;to-node and bounding ellipse
				    (:r (o-formula 
					 (sqrt (+ (expt (* (cos (gvl :theta))
							   (/ (gvl 
							       :to-node 
							       :frame 
							       :width) 
							      2)) 2)
						  (expt (* *halfnodeheight* 
							   (sin (gvl :theta))) 
							2)))))
				    (:x1 (o-formula (+ (gvl :from-node :left)
						       (floor (gvl 
							       :from-node 
							       :frame 
							       :width) 
							      2))))
				    (:y1 (o-formula (+ (gvl :from-node :top) 
						       *halfnodeheight*)))
				    (:x2 (o-formula 
					  (truncate (+ (gvl :to-node :left)
						       (/ (gvl 
							   :to-node 
							   :frame 
							   :width) 
							  2)
						       (* (- *halfnodeheight*)
							  (/ (gvl 
							      :to-node 
							      :frame 
							      :width) 
							     2)
							  (sin (gvl :theta)) 
							  (/ (gvl :r)))))))
				    (:y2 (o-formula 
					  (truncate (+ (gvl :to-node :top) 
						       *halfnodeheight*
						       (* (- *halfnodeheight*)
							  (/ (gvl 
							      :to-node 
							      :frame 
							      :width) 
							     2)
							  (cos (gvl :theta)) 
							  (/ (gvl :r)))))))
				    (:label1 "  ")) ;Label is initially blank
		   (opal:add-component (g-value display-window :inner-aggregate)
				       (eval obj)
				       :where :back)
					;>>> Put arc into this slot of the
					;>>> display window 
		   (push (eval obj) (g-value display-window 
					     :temporary-nodes-and-arcs-on-display))
					;>>> Update all relevant references
		   (push (list (eval obj) from-node) 
			 (g-value to-node :parent-node))
		   (push (list (eval obj) to-node)
			 (g-value from-node :children))
		   (eval obj))))))


; ==================================================================================
;
; CREATE_BLANK_DOUBLE_ARC
; -----------------
;
;          Arguments         :   <two-point-interactor>, <points-list>
;
;          Returns           :   A garnet-double-arc anchored at two existing garnet
;                                nodes. 
;
;          Description       :   This function is given a points list of the form
;                                (x1 y1 x2 y2) and creates a new double-arc that 
;                                becomes attached to the two nodes that exist at 
;                                those coordinates.   Esentially the same as
;                                as create_blank_arc.
;
;          Dependencies      :   Called by the two-point-interactor
;                                create_blank_double_arc_inter (xginseng.lisp)
;
; ==================================================================================

(defun create_blank_double_arc (inter points-list)
  (declare (ignore inter))
  (when points-list
					;>>> First see if the two points are above
					;>>> two existing garnet nodes.  If not,
					;>>> beep an error message and exit
	(let ((obj (intern (symbol-name (gensym "DOUBLE-ARC")) 'xginseng))
	      ;;get the identity of the from node
	      (from-node (opal:point-to-component 
			  (g-value display-window :inner-aggregate)
			  (first points-list)
			  (second points-list) :type node))
	      ;;get the identity of the to node
	      (to-node (opal:point-to-component 
			(g-value display-window :inner-aggregate)
			(third points-list)
			(fourth points-list) :type node)))
	  (if (or (null from-node)
		  (null to-node)
		  (eq from-node to-node))
	      (inter:beep)
	    (block false-clause-make-arc
		   (create-instance obj double-arc
				    (:from-node from-node)
				    (:to-node to-node)
					;>>> Geometry calculations by
					;>>> Martin Zaidel
				    (:quadrant (o-formula 
						(if (> (gvl :from-node :top) 
						       (gvl :to-node :top))
						    (if (< (gvl :from-node :left) 
							   (gvl :to-node :left))
					;>>> dest is in source's 1st quadrant
							1
					;>>> else dest is in its 2nd quadrant
						      2)
						  (if (> (gvl :from-node :left) 
							 (gvl :to-node :left))
					;>>> dest is in source's 3rd quadrant
						      3
					;>>> else dest is in its 4th quadrant
						    4))))
				    (:left (o-formula 
					    (case (gvl :quadrant)
						  ((1 4) 
						   (+ 
						    (gvl :from-node :left) 
						    (floor 
						     (gvl :from-node :width) 2)))
						  (t (+ 
						      (gvl :to-node :left) 
						      (floor 
						       (gvl :to-node :width) 2))))))
				    (:top (o-formula 
					   (case (gvl :quadrant)
						 ((1 2) 
						  (+ 
						   *halfnodeheight* 
						   (gvl :to-node :top)))
						 (t (+ 
						     *halfnodeheight* 
						     (gvl :from-node :top))))))
				    (:right (o-formula 
					     (case (gvl :quadrant)
						   ((1 4) 
						    (+ 
						     (gvl :to-node :left) 
						     (floor 
						      (gvl :to-node :width) 2)))
						   (t (+ 
						       (gvl :from-node :left) 
						       (floor 
							(gvl :from-node :width) 2))))))
				    (:bottom (o-formula 
					      (case (gvl :quadrant)
						    ((1 2) 
						     (+ 
						      *halfnodeheight* 
						      (gvl :from-node :top)))
						    (t (+ 
							*halfnodeheight* 
							(gvl :to-node :top))))))
				    (:width (o-formula 
					     (abs (- (+ 
						      (gvl :from-node :left) 
						      (floor 
						       (gvl :from-node :width) 2))
						     (+ (gvl :to-node :left) 
							(floor 
							 (gvl :to-node :width) 2))))))
				    (:height (o-formula 
					      (abs (- 
						    (gvl :from-node :top) 
						    (gvl :to-node :top)))))
				    (:target-width (o-formula 
						    (gvl :to-node :width)))
				    (:top-label "    ")
				    (:bottom-label "    "))
		   (opal:add-component (g-value display-window :inner-aggregate)
				       (eval obj)
				       :where :back)
		   (push (eval obj) (g-value display-window 
					     :temporary-nodes-and-arcs-on-display))
		   (push (list (eval obj) from-node)
			 (g-value to-node :parent-node))
		   (push (list (eval obj) to-node)
			 (g-value from-node :children))
		   (eval obj))))))








; ==================================================================================
;
; XGINSENG-MAKE-SNEPS-COMMAND
; ---------------------------
;
;          arguments         :   <garnet-node>
;
;          returns           :   snepsul command corresponding to the node
;
;          description       :   this function recursively builds up a snepsul
;                                command and enters it into the :sneps-command
;                                slot of the garnet-node
;
;          dependencies      :   no other xginseng functions called
;
; ==================================================================================

(defun xginseng-make-sneps-command (a-node)
  "Takes a Garnet node as input and returns its correspoding 
   SNePS command"
  (let ((sneps-node (g-value a-node :sneps-node))) ;pointer to SNePS node, if exists
    (if sneps-node ;if SNePS node exists, then its symbol is the command
	(s-value a-node :sneps-command sneps-node)
      (cond ((null (g-value a-node :children)) ;if it has no children
	     (s-value a-node :sneps-command ; then its symbol is its command
		      (g-value a-node :name-symbol)))
	    (t 
	     (let ((descendant-symbols nil)) ;Otherwise look at its children
	       (dolist (cable (g-value a-node :children) nil) 
		       (let ((child-node (second cable)) 
			     (child-arc (first cable)))
					;>>> Recursively build command of children
			 (xginseng-make-sneps-command child-node) 
			 (push (g-value child-node :sneps-command)
			       descendant-symbols)
					;>>> Case of a double-arc
			 (if (equalp (g-value child-arc :prototype) 'double-arc)
			     (progn 
			       (push (g-value child-arc :top-label-symbol)
				     descendant-symbols)
			       (push (g-value child-node :sneps-command)
				     descendant-symbols)
			       (push (g-value child-arc :bottom-label-symbol)
				     descendant-symbols))
			   (push (g-value child-arc :label1-symbol)
				 descendant-symbols))))
					;>>> Prepend an 'assert' if the node is 
					;>>> marked as such
	       (if (g-value a-node :assert-node-p)
		   (s-value a-node :sneps-command
			    (push 'assert descendant-symbols))
					;>>> Or a 'build'
		 (s-value a-node :sneps-command
			  (push 'build descendant-symbols)))))))))



; ==================================================================================
;
; ASSERT_BUILD
; ------------
;
;          Arguments         :   <button-interactor>, <garnet-node>
;
;          Returns           :   None
;
;          Description       :   This function set/unsets the :assert-node-p flag
;                                in <garnet-node> each time the :F3 key is pressed
;                                on the keyboard and the cursor is over the node.
;                                Also, the function inserts a '!' into the display
;                                to alert the user that the node is an assert 
;                                instead of a build.
;           
;          Dependencies      :   Called by the button-interactor
;                                assert-build-toggle-inter (xginseng.lisp)
;
; ==================================================================================


(defun assert_build (inter node)
  (declare (ignore inter))
  (if (g-value node :assert-node-p)
      (block true-clause
	     (s-value node :assert-node-p nil)
	     (s-value node :name "  "))
    (block false-clause
	   (s-value node :assert-node-p t)
	   (s-value node :name " !"))))



; ==================================================================================
;
; XGINSENG-SNEPS-EXECUTE & XGINSENG-SNEPS-EXECUTE-DIRECT
; ------------------------------------------------------
;
;          Arguments         :   A legal snepsul command
;
;          Returns           :   Return value generated by a snepsul command
;
;          Description       :   Both these functions use the with-snepsul
;                                package written by Hans Chalupsky
;
;          Dependencies      :   Called by define_relations and enter_commands
;
; ==================================================================================

(defun xginseng-sneps-execute (command)
  #3!(~command))

(defun xginseng-sneps-execute-direct (command)
  #2!(~command))

(defun xginseng-sneps-execute-direct_mult (node)
  #2!((describe ~node)))


; ==================================================================================
;
; DEFINE_RELATIONS
; ----------------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Description       :   This function assembles all the labels from 
;                                all new garnet arcs generated by the user, removes
;                                those that are already SNePS relations, removes
;                                duplicates, executes sneps:define, and writes
;                                the command to the log file "sneps-command_log.lisp"
;                                
;
;          Dependencies      :   Called by xginseng-sneps-execute
;
; ==================================================================================

(defun define_relations ()
  (let ((all-temporary-structures ;>>>All garnet nodes and arcs that have no
	 ;>>> corresponding SNePS structure associated with them
	 (g-value display-window :temporary-nodes-and-arcs-on-display))
	(arc-labels nil)
	(relations-already-defined 
	 (xginseng-sneps-execute '(snepsul::* 'relations))))
    ;>>> Assemble all relation labels
    (dolist (structure all-temporary-structures nil)
	    (if (eq (g-value structure :is-a) arc)
		(push (intern (symbol-name (g-value structure :label1-symbol))
			      'snepsul)
		      arc-labels))
	    (if (eq (g-value structure :is-a) double-arc)
		(push (intern (symbol-name (g-value structure :top-label-symbol))
			      'snepsul)
		      arc-labels))
	    (if (eq (g-value structure :is-a) double-arc)
		(push (intern (symbol-name (g-value structure :bottom-label-symbol))
			      'snepsul)
		      arc-labels)))
    (let ((relations-to-define ; Relations to be included in command
	   (remove-duplicates
	    (set-difference 
	     arc-labels
	     relations-already-defined))))
      (if relations-to-define
	  (block nil
		 (let ((final-command 
			(cons 'sneps:define relations-to-define)))
					;>>> Execute
		   (xginseng-sneps-execute final-command)))))))


; ==================================================================================
;
; ENTER_COMMANDS
; --------------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Description       :   Main function of input package.   Checks for
;                                correctness of garnet structure, generates and
;                                executes top level commands for all top level nodes,
;                                and restructures display structure to correspond
;                                to that of now existing SNePS network
;                                
;                                
;
;          Dependencies      :   Calls check_if_legal_sneps, define_relations,
;                                xginseng-sneps-execute, xginseng-make-sneps-command,
;                                and merge_input_and_output
;
; ==================================================================================

(defun enter_commands (toolkit-obj button-panel-item)
  (declare (ignore toolkit-obj button-panel-item))
  (clean_up_dialogue_window)
  (s-value main-button-panel :value-obj nil)
  (s-value main-button-panel :value nil)
  (create-instance 'user-info-on-network-entry opal:multi-text
		   (:left 10)
		   (:top 10)
		   (:string "Please refer to your Lisp invocation for any
relation definitions."))
  (opal:add-component dialogue-aggregate user-info-on-network-entry)
  (opal:update dialogue-window)
  (if (and (check_if_legal_sneps) ;Check for major syntax errors 
	   (define_relations_2)) ;make sure user wants those relations
      (block true-clause-legal-sneps
	     (let ((all-temporary-structures 
		    (reverse (g-value 
			      display-window 
			      :temporary-nodes-and-arcs-on-display)))
		   (top-level-sneps-node nil))
					;>>> Look for top-level nodes, build 
					;>>> commands, and execute
	       (dolist (structure all-temporary-structures nil)
		       (if (and (equalp (g-value structure :prototype) 'node)
				(null (g-value structure :parent-node)))
			   (block true-clause-top-level-molecular-node
				  (setf top-level-sneps-node ;set pointer to new 
					(sneps:node          ;sneps node
					 (first (xginseng-sneps-execute 
						 (xginseng-make-sneps-command 
						  structure)))))
					;Set slot of garnet-node to point to new
					;SNePS node
				  (s-value structure :sneps-node top-level-sneps-node)
					;Set :name slot of garnet-node to be that of
					;the SNePS node
				  (s-value structure :name
					   (format nil 
						   "~a" 
						   (g-value 
						    structure 
						    :sneps-node))))))
					;>>> Execute SNePS commands of lower level
					;>>> garnet-nodes in order to find
					;>>> pointers to corresponding SNePS nodes
	       (dolist (structure all-temporary-structures nil)
		       (if (and (equalp (g-value structure :prototype) 'node)
				(null (g-value structure :sneps-node)))
			   (let ((sneps-command (g-value structure :sneps-command)))
			     (cond ((listp sneps-command) ;the node has children
				    (block true-clause-non-top-molecular-node
					   (s-value structure :sneps-node
						    (sneps:node
						     (first 
						      (xginseng-sneps-execute-direct
						       sneps-command))))
					   (s-value structure :name
						    (format nil 
							    "~A" 
							    (g-value 
							     structure 
							     :sneps-node)))
					   (s-value structure :name-symbol
						    (intern 
						     (g-value structure :name) 
						     'xginseng))))
				   ((symbolp sneps-command) ;node is base node
				    (block true-clause-base-node 
					   (s-value structure :sneps-node
						    (sneps:node sneps-command))
					   (s-value structure :name 
						    (symbol-name sneps-command))))))))
	       (merge_input_and_output) ;>>> Restructure garnet-nodes
	       (clean_up_dialogue_window)
	       (s-value main-button-panel :value-obj nil)
	       (s-value main-button-panel :value nil)
	       (s-value main-button-panel-2 :value-obj nil)
	       (s-value main-button-panel-2 :value nil))))
  (clean_up_dialogue_window)
  (s-value main-button-panel :value-obj nil)
  (s-value main-button-panel :value nil)
  (s-value main-button-panel-2 :value-obj nil)
  (s-value main-button-panel-2 :value nil))




; ==================================================================================
;
; MERGE_INPUT_AND_OUTPUT
; ---------------------------
;
;          Arguments         :   None
;
;          Returns           :   None
;
;          Description       :   Restructures the garnet-node network immediately
;                                after its commands have been executed and the
;                                garnet-nodes have had their pointers set to their 
;                                SNePS nodes.  The old nodes act as prototypes 
;                                for new nodes that will be named according to 
;                                their respective SNePS nodes.
;                                
;
;          Dependencies      :   Called by enter_commands
;
; ==================================================================================

(defun merge_input_and_output ()
  (let ((all-temporary-structures 
	 (g-value display-window :temporary-nodes-and-arcs-on-display)))
					;>>> Find a nodesand see if there exits
					;>>> another garnet node with the same name.  
					;>>> If not, create a new node with that name
					;>>> using present node as a prototype
    (dolist (structure all-temporary-structures nil)
	    (if (equalp (g-value structure :prototype) 'node)
		(let ((new_node_symbol (intern 
					(string-right-trim '(#\!) 
							   (g-value structure :name)) 
					'xginseng))
		      (new_node nil)) ;Pointer to new node to be created
		  (if (member new_node_symbol ;See if there exists another garnet-node
			      (mapcar         ;with the same name
			       #'(lambda (x) (find-symbol (name-for-schema x) 'xginseng))
			       (g-value display-window :nodes-on-display)))
		      (s-value structure ;If so, then set :alter-ego slot of
			       :alter-ego ;current node pointing to the original
			       (eval (find-symbol new_node_symbol 'xginseng)))
		    (block false-clause-duplicate-node
					;>>> Not a duplicate so create a new node
			   (setf new_node (create-instance new_node_symbol structure))
			   (opal:add-component 
			    (g-value display-window :inner-aggregate) new_node)
			   (push new_node (g-value display-window :nodes-on-display))
			   (setf (g-value display-window :nodes-on-display)
				 (remove structure (g-value display-window :nodes-on-display)))
					;>>> Reset arcs that were associated with
					;>>> the old node so they are now 
					;>>> associated with the new one
			   (dolist (children (g-value structure :children) nil)
				   (s-value (first children) :from-node new_node))
			   (dolist (parent-nodes (g-value structure :parent-node) nil)
				   (s-value (first parent-nodes) :to-node new_node))
					;>>> Set alter-ego of new node
					;>>> pointing to itself
					;>>> This slot always points to the
					;>>> garnet-node that is named
					;>>> after its respective SNePS
					;>>> node
			   (s-value structure :alter-ego new_node))))))
					;>>> Set pointers of all arcs to point
					;>>> to the proper nodes
    (dolist (structure all-temporary-structures nil)
	    (if (equalp (g-value structure :prototype) 'node)
		(block true-clause-isa-node-set-arcs
		       (dolist (children 
				(reverse (g-value structure :children)) nil)
			       (s-value (first children) 
					:to-node (g-value (second children) :alter-ego)) 
			       (push (list (first children)
					   (g-value (second children) :alter-ego))
				     (g-value 
				      (g-value structure 
					       :alter-ego) 
				      :down-cables)))
		       (dolist (parent-nodes 
				(reverse (g-value structure :parent-node)) nil)
			       (s-value (first parent-nodes) 
					:from-node (g-value (second parent-nodes) :alter-ego)) 			  
			       (push (list (first parent-nodes)
					   (g-value (second parent-nodes) :alter-ego))
				     (g-value 
				      (g-value structure 
					       :alter-ego) 
				      :up-cables))))))
					;>>> Remove prototype nodes from the display
					;>>> aggreate
    (dolist (structure all-temporary-structures nil)
	    (if (equalp (g-value structure :prototype) 'node)
		(opal:remove-component 
		 (g-value display-window :inner-aggregate) structure)))
					;>>> Remove all temporary structures
    (s-value display-window :temporary-nodes-and-arcs-on-display nil)))


; ==================================================================================
;
; CHECK_IF_LEGAL_SNEPS
; --------------------
;
;          Arguments         :   None
;
;          Returns           :   t if legal struture, nil otherwise
;
;          Description       :   Checks garnet structure for isolated nodes,
;                                non-asserted top-level nodes, missing relation
;                                labels, and missing node labels.  Immediatetely
;                                returns nil if any of these conditions holds.
;                                alerts the user by beeping and flashing the
;                                offending object (flashing is part of the 
;                                garnet-debug package).
;
;          Dependencies      :   Called by enter_commands
;
; ==================================================================================

(defun check_if_legal_sneps ()
  (let ((all-temporary-structures 
	 (g-value display-window :temporary-nodes-and-arcs-on-display))
	(all-clear-flag t)) ;Assume things are legal
					;>>> Look at all temporary structures
    (dolist (structure all-temporary-structures all-clear-flag)
	    (if (schema-p structure)
		(cond ((and (equalp (g-value structure :prototype) 'node)
			    (null (g-value structure :parent-node)) ;Case of isolated 
			    (null (g-value structure :children))    ;node 
			    (or (null (find-symbol ;or unlabeled node
				       (string-upcase 
					(g-value structure :name)) 'xginseng))
				(not (boundp 
				      (find-symbol 
				       (string-upcase 
					(g-value structure :name)) 'xginseng)))))
		       (inter:beep)
		       (garnet-debug:flash structure)
		       (setq all-clear-flag nil))
		      ((and (equalp (g-value structure :prototype) 'arc)
			    (equalp (length      ;Case of unlabeled arc
				     (string-trim 
				      '(#\space) 
				      (g-value structure 
					       :label-box :thelabel :string))) 0))
		       (inter:beep)
		       (garnet-debug:flash (g-value structure :label-box))
		       (setq all-clear-flag nil))
		      ((and (equalp (g-value structure :prototype) 'double-arc)
			    (equalp (length      ;Case of unlabeled double-arc
				     (string-trim 
				      '(#\space) 
				      (g-value structure 
					       :top-label-box :thelabel :string))) 0))
		       (inter:beep)
		       (garnet-debug:flash (g-value structure :top-label-box))
		       (setq all-clear-flag nil))
		      ((and (equalp (g-value structure :prototype) 'double-arc)
			    (equalp (length 
				     (string-trim 
				      '(#\space) 
				      (g-value structure 
					       :bottom-label-box :thelabel :string))) 0))
		       (inter:beep)
		       (garnet-debug:flash (g-value structure :bottom-label-box))
		       (setq all-clear-flag nil))
		      ((and (equalp (g-value structure :prototype) 'node)
			    (null (g-value structure :children))
			    (equalp (length 
				     (string-trim 
				      '(#\space) 
				      (g-value structure :text :string))) 0))
		       (inter:beep)
		       (garnet-debug:flash structure)
		       (setq all-clear-flag nil))
		      ((and (equalp (g-value structure :prototype) 'node)
			    (null (g-value structure :parent-node)) ;un-asserted top-
			    (null (g-value structure :assert-node-p)));level node
		       (inter:beep)
		       (garnet-debug:flash structure)
		       (setq all-clear-flag nil))
		      (t (values)))))))
  

; ==================================================================================
;
; DESTROY_GARNET_OBJECT 
; ---------------------
;
;          Arguments         :   <button-interactor>, <garnet-object>
;
;          Returns           :   nil
;
;          Description       :   Deletes a garnet node or arc Ithat has no sneps
;                                node associated with it) when the user clicks 
;                                control-shift-leftdown.  If a node is deleted 
;                                then all arcs entering or leaving it are also
;                                destroyed.
;
;          Dependencies      :   Called by the button-interactor
;                                node-or-arc-deletor (xginseng.lisp).
;                                Calls destroy_arc_references_in_child_node,
;                                and destroy_arc_references_in_parent_node.
;
; ==================================================================================


(defun destroy_garnet_object (inter obj)
  (declare (ignore inter))
  (let ((node-to-be-killed (opal:point-to-component 
			    (g-value display-window :inner-aggregate)
			    (inter:event-x inter:*current-event*)
			    (inter:event-y inter:*current-event*)
			    :type
			    (eval node))))
    (if node-to-be-killed
	(setf obj node-to-be-killed))
    (cond ((equal (g-value obj :prototype) 'node)
	   (progn
	     (dolist (up-cable (g-value obj :parent-node))
		     (let ((arc (first up-cable))
			   (destination-node (second up-cable)))
		       (destroy_arc_references_in_parent_node arc destination-node)
		       (opal:destroy arc)))
	     (dolist (up-cable (g-value obj :up-cables))
		     (let ((arc (first up-cable))
			   (destination-node (second up-cable)))
		       (destroy_arc_references_in_parent_node arc destination-node)
		       (opal:destroy arc)))
	     (dolist (down-cable (g-value obj :children))
		     (let ((arc (first down-cable))
			   (destination-node (second down-cable)))
		       (destroy_arc_references_in_child_node arc destination-node)
		       (opal:destroy arc)))
	     (dolist (down-cable (g-value obj :down-cables))
		     (let ((arc (first down-cable))
			   (destination-node (second down-cable)))
		       (destroy_arc_references_in_child_node arc destination-node)
		       (opal:destroy arc)))
	     (setf (g-value display-window :nodes-on-display)
		   (remove obj (g-value display-window :nodes-on-display)))
	     (opal:remove-component 
	      (g-value display-window :inner-aggregate)
	      obj)
	     (opal:destroy obj)
	     (s-value display-window :nodes-on-display
		      (remove-if-not
		       #'(lambda (x) (schema-p x))
		       (g-value display-window :nodes-on-display)))
	     (s-value display-window :temporary-nodes-and-arcs-on-display
		      (remove-if-not
		       #'(lambda (x) (schema-p x))
		       (g-value display-window :temporary-nodes-and-arcs-on-display)))))
	  (t
	   (destroy_arc_references_in_parent_node 
	    obj
	    (g-value obj :from-node))
	   (destroy_arc_references_in_child_node 
	    obj
	    (g-value obj :to-node))
	   (opal:destroy obj)
	   (s-value display-window :nodes-on-display
		    (remove-if-not
		     #'(lambda (x) (schema-p x))
		     (g-value display-window :nodes-on-display)))
	   (s-value display-window :temporary-nodes-and-arcs-on-display
		    (remove-if-not
		     #'(lambda (x) (schema-p x))
		     (g-value display-window :temporary-nodes-and-arcs-on-display)))))))

(defun destroy_arc_references_in_child_node (arc destination-node)
  (dolist (a_cable (g-value destination-node :parent-node))
	  (if (equalp (first a_cable) arc)
	      (s-value destination-node :parent-node
		       (remove a_cable (g-value destination-node :parent-node)))))
  (dolist (a_cable (g-value destination-node :up-cables))
	  (if (and a_cable
		   (equalp (first a_cable) arc))
	      (s-value destination-node :up-cables
		       (remove a_cable (g-value destination-node :up-cables))))))



(defun destroy_arc_references_in_parent_node (arc destination-node)
  (dolist (a_cable (g-value destination-node :children))
	  (if (equalp (first a_cable) arc)
	      (s-value destination-node :children
		       (remove a_cable (g-value destination-node :children)))))
  (dolist (a_cable (g-value destination-node :down-cables))
	  (if (equalp (first a_cable) arc)
	      (s-value destination-node :down-cables
		       (remove a_cable (g-value destination-node :down-cables))))))




(setf moving-oval
      (create-instance 'moving-oval opal:oval
		       (:box (list 0 0 0 0))
		       (:left (o-formula (first (gvl :box))))
		       (:top (o-formula (second (gvl :box))))
		       (:width *halfnodeheight*)
		       (:height *nodeheight*)))


(create-instance 'moving-arrow-line opal:line
		 (:points (list 0 0 20 20))
		 (:x1 (o-formula (first (gvl :points))))
		 (:y1 (o-formula (second (gvl :points))))
		 (:x2 (o-formula (third (gvl :points))))
		 (:y2 (o-formula (fourth (gvl :points)))))


(defun erase-node (a-node)
  (if (equal (g-value a-node :prototype) 'node)
      (let ((command (list 'erase (intern (name-for-schema a-node) 'xginseng)))
	    (all-nodes (g-value display-window :nodes-on-display)))
	(xginseng-sneps-execute command)
	(dolist (a-node all-nodes)
		(s-value a-node :sneps-node 
			 (sneps:node (string-trim '(#\!) (name-for-schema a-node)))))
	(dolist (a-node (g-value display-window :nodes-on-display))
		(if (and (null (g-value a-node :sneps-node))
			 (kr::schema-name a-node))
		    (destroy_node a-node))))))


;;	(dolist (a-node all-nodes)
;;		(if (null (sneps:node (string-trim '(#\!) (name-for-schema a-node))))
;;		    (destroy_garnet_object nil a-node)))
;;	(dolist (a-node all-nodes)
;;		(if 
;;		    (destroy_garnet_object nil a-node)))
;;	



(defun assert_a_node (a-node)
  (let ((command (list 'sneps:|!| (g-value a-node :name-symbol))))
    (xginseng-sneps-execute command)
    (s-value a-node :sneps-node (sneps:node (g-value a-node :name-symbol)))
    (s-value a-node :name (format nil "~a" (g-value a-node :sneps-node)))))


(defun recover ()
  (if (boundp 'xginseng::display-window)
      (opal:destroy xginseng::display-window))
  (if (boundp 'xginseng::dialogue-window)
      (opal:destroy xginseng::dialogue-window))
  (if (boundp 'xginseng::button-window)
      (opal:destroy xginseng::button-window))
  (if (boundp 'xginseng::input-button-window)
      (opal:destroy xginseng::input-button-window))
  (if (boundp 'xginseng::help-window)
      (opal:destroy xginseng::help-window))
  (xginseng::xginseng))



(defun destroy_node (obj)
  (cond ((equal (g-value obj :prototype) 'node)
	 (progn
	   (dolist (up-cable (g-value obj :parent-node))
		   (let ((arc (first up-cable))
			 (destination-node (second up-cable)))
		     (destroy_arc_references_in_parent_node arc destination-node)
		     (if (schema-p arc)
			 (opal:destroy arc))))
	   (dolist (up-cable (g-value obj :up-cables))
		   (let ((arc (first up-cable))
			 (destination-node (second up-cable)))
		     (destroy_arc_references_in_parent_node arc destination-node)
		     (if (schema-p arc)
			 (opal:destroy arc))))
	   (dolist (down-cable (g-value obj :children))
		   (let ((arc (first down-cable))
			 (destination-node (second down-cable)))
		     (destroy_arc_references_in_child_node arc destination-node)
		     (if (schema-p arc)
			 (opal:destroy arc))))
	   (dolist (down-cable (g-value obj :down-cables))
		   (let ((arc (first down-cable))
			 (destination-node (second down-cable)))
		     (destroy_arc_references_in_child_node arc destination-node)
		     (if (schema-p arc)
			 (opal:destroy arc))))
	   (setf (g-value display-window :nodes-on-display)
		 (remove obj (g-value display-window :nodes-on-display)))
	   (opal:remove-component 
	    (g-value display-window :inner-aggregate)
	    obj)
	   (opal:destroy obj)))
	(t nil)))

(defvar *query-user* nil)

(defun define_relations_2 ()
  (let ((all-temporary-structures
	 (g-value display-window :temporary-nodes-and-arcs-on-display))
	(all-clear-flag t))
    (dolist (structure all-temporary-structures)
	    (if (equal (first (g-value structure :is-a)) arc)
		(let ((relation-name 
		       (intern (symbol-name (g-value structure :label1-symbol)) 'snepsul))
		      (relations-already-defined (xginseng-sneps-execute '(snepsul::* 'relations))))
		  (if (not (member relation-name relations-already-defined))
		      (progn
			(setf *query-user*
			      (concatenate 'string "Do you want to define "
					   (format nil "~a" relation-name)
					   "?"))
			(inter:beep)
			(if (y-or-n-p *query-user*)
			    (let ((command (list 'define relation-name)))
			      (xginseng-sneps-execute command))
			  (setf all-clear-flag nil)))))))
    (dolist (structure all-temporary-structures)
	    (if (equal (first (g-value structure :is-a)) double-arc)
		(let ((top-relation-name 
		       (intern (symbol-name (g-value structure :top-label-symbol)) 'snepsul))
		      (bottom-relation-name 
		       (intern (symbol-name (g-value structure :bottom-label-symbol)) 'snepsul))
		      (relations-already-defined (xginseng-sneps-execute '(snepsul::* 'relations))))
		  (if (not (member top-relation-name relations-already-defined))
		      (progn
			(setf *query-user*
			      (concatenate 'string "Do you want to define "
					   (format nil "~a" top-relation-name)
					   "?"))
			(inter:beep)
			(if (y-or-n-p *query-user*)
			    (let ((command (list 'define top-relation-name)))
			      (xginseng-sneps-execute command))
			  (setf all-clear-flag nil))))
		  (if (not (member bottom-relation-name relations-already-defined))
		      (progn
			(setf *query-user*
			      (concatenate 'string "Do you want to define "
					   (format nil "~a" bottom-relation-name)
					   "?"))
			(inter:beep)
			(if (y-or-n-p *query-user*)
			    (let ((command (list 'define bottom-relation-name)))
			      (xginseng-sneps-execute command))
			  (setf all-clear-flag nil)))))))
    (if all-clear-flag
	(format t "Creating a SNePS network...~%~%")
      (format t "Please re-edit your network.~%~%"))
    all-clear-flag))



(defun create_universal_quantifier (inter points-list)
  (let* ((top-node-coordinates (list (first points-list)
				     (- (second points-list) 100)
				     (third points-list)
				     (fourth points-list)))
	 (var-node-coordinates (list (- (first points-list) 100)
				     (second points-list)
				     (third points-list)
				     (fourth points-list)))
	 (ant-node-coordinates (copy-list points-list))
	 (cq-node-coordinates (list (+ (first points-list) 100)
				    (second points-list)
				    (third points-list)
				    (fourth points-list)))
	 (top-node (create_blank_node inter top-node-coordinates))
	 (var-node (create_blank_node inter var-node-coordinates))
	 (ant-node (create_blank_node inter ant-node-coordinates))
	 (cq-node (create_blank_node inter cq-node-coordinates))
	 (arc-one (create_blank_arc inter (list (opal:center-x top-node)
						(opal:center-y top-node)
						(opal:center-x var-node)
						(opal:center-y var-node))))
	 (arc-two (create_blank_arc inter (list (opal:center-x top-node)
						(opal:center-y top-node)
						(opal:center-x ant-node)
						(opal:center-y ant-node))))
	 (arc-three (create_blank_arc inter (list (opal:center-x top-node)
						  (opal:center-y top-node)
						  (opal:center-x cq-node)
						  (opal:center-y cq-node)))))
    (assert_build nil top-node)
    (s-value (eval arc-one) :label1 "forall")
    (s-value (eval arc-two) :label1 "&ant")
    (s-value (eval arc-three) :label1 "cq")))





(defun create_numerical_entailment_frame (inter points-list)
  (let* ((top-node-coordinates (list (first points-list)
				     (- (second points-list) 100)
				     (third points-list)
				     (fourth points-list)))
	 (var-node-coordinates (list (- (first points-list) 100)
				     (second points-list)
				     (third points-list)
				     (fourth points-list)))
	 (ant-node-coordinates (copy-list points-list))
	 (cq-node-coordinates (list (+ (first points-list) 100)
				    (second points-list)
				    (third points-list)
				    (fourth points-list)))
	 (top-node (create_blank_node inter top-node-coordinates))
	 (var-node (create_blank_node inter var-node-coordinates))
	 (ant-node (create_blank_node inter ant-node-coordinates))
	 (cq-node (create_blank_node inter cq-node-coordinates))
	 (arc-one (create_blank_arc inter (list (opal:center-x top-node)
						(opal:center-y top-node)
						(opal:center-x var-node)
						(opal:center-y var-node))))
	 (arc-two (create_blank_arc inter (list (opal:center-x top-node)
						(opal:center-y top-node)
						(opal:center-x ant-node)
						(opal:center-y ant-node))))
	 (arc-three (create_blank_arc inter (list (opal:center-x top-node)
						  (opal:center-y top-node)
						  (opal:center-x cq-node)
						  (opal:center-y cq-node)))))
    (assert_build nil top-node)
    (s-value (eval arc-one) :label1 "thresh")
    (s-value (eval arc-two) :label1 "&ant")
    (s-value (eval arc-three) :label1 "cq")))







(defun create_AndOr_frame (inter points-list)
  (let* ((top-node-coordinates (list (first points-list)
				     (- (second points-list) 150)
				     (third points-list)
				     (fourth points-list)))
	 (minmax-node-coordinates (list (- (first points-list) 225)
				     (second points-list)
				     (third points-list)
				     (fourth points-list)))
	 (min-node-coordinates (list (- (first points-list) 75)
				     (second points-list)
				     (third points-list)
				     (fourth points-list)))
	 (max-node-coordinates (list (+ (first points-list) 75)
				     (second points-list)
				     (third points-list)
				     (fourth points-list)))
	 (arg-node-coordinates (list (+ (first points-list) 225)
				    (second points-list)
				    (third points-list)
				    (fourth points-list)))
	 (top-node (create_blank_node inter top-node-coordinates))
	 (minmax-node (create_blank_node inter minmax-node-coordinates))
	 (min-node (create_blank_node inter min-node-coordinates))
	 (max-node (create_blank_node inter max-node-coordinates))
	 (arg-node (create_blank_node inter arg-node-coordinates))
	 (arc-one (create_blank_double_arc inter (list (opal:center-x top-node)
						(opal:center-y top-node)
						(opal:center-x minmax-node)
						(opal:center-y minmax-node))))
	 (arc-two (create_blank_arc inter (list (opal:center-x top-node)
						(opal:center-y top-node)
						(opal:center-x min-node)
						(opal:center-y min-node))))
	 (arc-three (create_blank_arc inter (list (opal:center-x top-node)
						  (opal:center-y top-node)
						  (opal:center-x max-node)
						  (opal:center-y max-node))))
	 (arc-four (create_blank_arc inter (list (opal:center-x top-node)
						  (opal:center-y top-node)
						  (opal:center-x arg-node)
						  (opal:center-y arg-node)))))
    (assert_build nil top-node)
    (s-value (eval arc-one) :top-label "min")
    (s-value (eval arc-one) :bottom-label "max")
    (s-value (eval arc-two) :label1 "min")
    (s-value (eval arc-three) :label1 "max")
    (s-value (eval arc-four) :label1 "arg")))






(defun create_thresh_frame (inter points-list)
  (let* ((top-node-coordinates (list (first points-list)
				     (- (second points-list) 150)
				     (third points-list)
				     (fourth points-list)))
	 (threshthreshmax-node-coordinates (list (- (first points-list) 225)
				     (second points-list)
				     (third points-list)
				     (fourth points-list)))
	 (thresh-node-coordinates (list (- (first points-list) 75)
				     (second points-list)
				     (third points-list)
				     (fourth points-list)))
	 (threshmax-node-coordinates (list (+ (first points-list) 75)
				     (second points-list)
				     (third points-list)
				     (fourth points-list)))
	 (arg-node-coordinates (list (+ (first points-list) 225)
				    (second points-list)
				    (third points-list)
				    (fourth points-list)))
	 (top-node (create_blank_node inter top-node-coordinates))
	 (threshthreshmax-node (create_blank_node inter threshthreshmax-node-coordinates))
	 (thresh-node (create_blank_node inter thresh-node-coordinates))
	 (threshmax-node (create_blank_node inter threshmax-node-coordinates))
	 (arg-node (create_blank_node inter arg-node-coordinates))
	 (arc-one (create_blank_double_arc inter (list (opal:center-x top-node)
						(opal:center-y top-node)
						(opal:center-x threshthreshmax-node)
						(opal:center-y threshthreshmax-node))))
	 (arc-two (create_blank_arc inter (list (opal:center-x top-node)
						(opal:center-y top-node)
						(opal:center-x thresh-node)
						(opal:center-y thresh-node))))
	 (arc-three (create_blank_arc inter (list (opal:center-x top-node)
						  (opal:center-y top-node)
						  (opal:center-x threshmax-node)
						  (opal:center-y threshmax-node))))
	 (arc-four (create_blank_arc inter (list (opal:center-x top-node)
						  (opal:center-y top-node)
						  (opal:center-x arg-node)
						  (opal:center-y arg-node)))))
    (assert_build nil top-node)
    (s-value (eval arc-one) :top-label "thresh")
    (s-value (eval arc-one) :bottom-label "threshmax")
    (s-value (eval arc-two) :label1 "thresh")
    (s-value (eval arc-three) :label1 "threshmax")
    (s-value (eval arc-four) :label1 "arg")))







    
    




