;; -*- Mode: Lisp; Syntax: Common-Lisp; Package: SNEPSUL; Base:10 -*-

;; Copyright (C) 1984--2010
;; Research Foundation of State University of New York

;; Version: $Id: driver.lisp,v 1.3 2010/09/04 02:31:58 mwk3 Exp $

;; This file is part of SNePS.

;; $BEGIN LICENSE$

;; 
;; The contents of this file are subject to the University at
;; Buffalo Public License Version 1.0 (the "License"); you may
;; not use this file except in compliance with the License. You
;; may obtain a copy of the License at http://www.cse.buffalo.
;; edu/sneps/Downloads/ubpl.pdf.
;; 
;; Software distributed under the License is distributed on an
;; "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, either express
;; or implied. See the License for the specific language gov
;; erning rights and limitations under the License.
;; 
;; The Original Code is SNePS 2.7.
;; 
;; The Initial Developer of the Original Code is Research Foun
;; dation of State University of New York, on behalf of Univer
;; sity at Buffalo.
;; 
;; Portions created by the Initial Developer are Copyright (C)
;; 2010 Research Foundation of State University of New York, on
;; behalf of University at Buffalo. All Rights Reserved.
;; 
;;  
;; 
;; 


;; $END LICENSE$


(in-package :snepsul)


;; Lisp side of the ARC/LISP communication package. Contains functions
;; that generate appropriate nextcommand.aml files and send them to
;; the communication directory where they will be read and executed
;; by the ARC command loop.

;; Author:   Hans Chalupsky
;; Created:  May 31, 1990
;; Modified: July 17, 1990

(defvar *interface-directory* "/u0/grads/hans/interface"
  "This is the name of the ARC/LISP communication directory viewed
from the machine where LISP runs on.")

;; Pathnames of various files viewed from the machine LISP is running on
(defvar *nextcom* (format nil "~a/nextcommand.aml" *interface-directory*))
(defvar *exit* (format nil "~a/exit" *interface-directory*))

;; The command-output file has to be defined in 2 steps because its name
;; is needed also on the machine ARC is running on.
(defvar *comout-name* "comout.log")
(defvar *comout* (format nil "~a/~a" *interface-directory* *comout-name*))


(defun wait-for-completion ()
  "Waits until the file nextcommand.aml gets deleted by the
AML command loop running in the ARC process. Returns once
nextcommand.aml does not exist anymore."
  (loop (unless (probe-file *nextcom*)
	  (return))
	;; give system some time to breathe
	(sleep 0.1)))

(defun execute-command (command-string)
  "Takes a COMMAND-STRING and writes a file nextcommand.aml into the
communication interface directory. The command loop running in the ARC
process waits for this file, executes it and deletes it. The output
generated by the command gets written to a command output (watch) file.
This allows to get results back into the LISP process."
  (wait-for-completion)
  (with-open-file (nextcom *nextcom* :direction :output)
    ;; nextcommand.aml takes as an argument the name of the communication
    ;; directory viewed from the ARC process, into which ARC will
    ;; write the comout file
    (format nextcom

	    "&args interface~%~
             &watch %interface%/~a~%~
             ~a~%~
             &watch &off~%~
             &return~%"

	    *comout-name* command-string)
    )
  (wait-for-completion))

(defun get-file-as-string (file)
  "Opens FILE and returns its contents as a string. Used to
read ARC command execution results back into the LISP process."
  (cond ((probe-file file)
	 (with-output-to-string (out)
	   (with-open-file (in file :direction :input)
	     (loop
	      (let ((line (read-line in nil :eof nil)))
		(cond ((eq line :eof)
		       (return))
		      (t (format out "~a~%" line))))))))
	(t "")))

(defun print-result (&optional (stream *standard-output*))
  "Prints contents of current command output file to STREAM."
  (format stream "~&~a" (get-file-as-string *comout*)))

(defun stop-arc ()
  "Creates an exit file in the communication directory which will cause
the arc command loop to terminate."
  (with-open-file (e *exit* :direction :output :if-exists nil)
    (format e " ")))

(defun arc-loop ()
  "Simple read/execute/print loop that reads ARC commands in LISP,
executes them in the ARC process and prints the results. Used for
testing of the interface. The command ^^ terminates the loop."
  (let ((command "")
	(prompt "Arc: "))
    #+ibcl(read-line)
    (loop
     (format t "~&~a" prompt)
     (setq command (read-line))
     (cond ((equal command "^^")
	    (stop-arc)
	    (return 'bye))
	   (t (execute-command command)
	      (format t "~a" (get-file-as-string *comout*)))))))

   



    
    




