function [H,F]=propagate(pM,class,k1,k2,alp,iter)
%
%  propagate: estimate the conditional probabilities of all the group nodes through label propagation 
%             
%  Input: 
%       pM-a matrix containing all the base models' outputs on the target set. 
%          Each column represent one model. You must place classification models
%           before clustering models.
%
%       k1-number of classification models, so columns 1:k1 in the matrix pM are the
%          results of classification models
%
%       k2-number of clustering models, so columns k1+1:k1+k2 in the matrix pM
%          are the results of clustering models
%
%       class-the number of classes in the problem
%
%       iter-number of iterations, usually set to 20
%
%       alp-parameter alpha in the paper, usually set to 0.4
%
%  Output:         
%
%       H-a matrix summarizing the group-example relationships. Each row
%         corresponds to an example whereas each column corresponds to a
%         group. If the example is contained in the group, the entry is 1,
%         otherwise it is 0.
%
%       F-a matrix containing the conditional probabilities of all the group
%         nodes. Each row represents one group, and column j is the
%         probability of each group belonging to class j
%
%
%

%initialization
[m,n]=size(pM);
sizeH=(k1+k2)*class;
H=zeros(m,sizeH);
%the initial probability of group nodes
Y=zeros(sizeH,class);

%get the group-example adjacency matrix from classifier outputs
%assign initial probability to group nodes from classifiers
for i=1:k1
    for j=1:class
        id=find(pM(:,i)==j);
        index=(i-1)*class+j;
        H(id,index)=1;
        Y(index,j)=1;
    end
end

%get the group-example adjacency matrix from clustering outputs
for i=1:k2
    for j=1:class
        id=find(pM(:,k1+i)==j);
        index=(k1+i-1)*class+j;
        H(id,index)=1;
    end
end  

%compute the similarity between every pair of group nodes
%the similarity is defined as the percentage of common members shared by
%the two groups
W=zeros(sizeH,sizeH);
for i=1:sizeH
    for j=i+1:sizeH
        W(i,j)=sum(H(:,i).*H(:,j))/sum(H(:,i)+H(:,j)-H(:,i).*H(:,j));
    end
end

%prepare for the propagation
W=W+W';
D=diag(sum(W));
K=inv(sqrt(D));
S=K*W*K;

%start with the initial labels
F=Y;

%iteratively propagate the conditional probability among group nodes
for i=1:iter
    F=alp*S*F+(1-alp)*Y;
end

%normalization
F=F./repmat(sum(F,2),1,class);
