/*
This file is licensed under the terms of the Globus Toolkit Public
License, found at http://www.globus.org/toolkit/download/license.html.
*/
package org.globus.ogsa.impl.samples.registry;

import org.globus.ogsa.GridServiceBase;
import org.globus.ogsa.GridContext;
import org.globus.ogsa.GridServiceCallback;
import org.globus.ogsa.GridServiceException;
import org.globus.ogsa.OperationProvider;
import org.globus.ogsa.ServiceProperties;
import org.globus.ogsa.ServiceData;
import org.globus.ogsa.utils.ServiceGroupUtils;
import org.globus.ogsa.utils.MessageUtils;

import org.gridforum.ogsi.OGSIServiceGridLocator;
import org.gridforum.ogsi.ServiceGroupRegistration;
import org.gridforum.ogsi.ExtendedDateTimeType;
import org.gridforum.ogsi.InfinityType;
import org.gridforum.ogsi.HandleType;
import org.gridforum.ogsi.ExtensibilityType;
import org.gridforum.ogsi.LocatorType;
import org.gridforum.ogsi.TerminationTimeType;
import org.gridforum.ogsi.holders.LocatorTypeHolder;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import javax.xml.namespace.QName;

/**
 * This class allows services to publish themselves to remote registries
 */
public class RegistryPublishProvider implements OperationProvider, 
                                                GridServiceCallback {
    static Log logger =
        LogFactory.getLog(RegistryPublishProvider.class.getName());
    private Thread thread;
    private OGSIServiceGridLocator locator = new OGSIServiceGridLocator();
    private LocatorType memberLocator;
    private ServiceGroupRegistration registry;

    public void initialize(GridServiceBase base)
            throws GridServiceException {
        String registryHandle = null;
        try {
            registryHandle = 
                (String)base.getProperty(ServiceProperties.REGISTRY);
            if (registryHandle == null) {
                logger.error("registry property must be set");
                return;
            }
            this.registry = 
                this.locator.getServiceGroupRegistrationPort(
                    new HandleType(registryHandle)
                );
            TerminationTimeType time = new TerminationTimeType();
            ExtendedDateTimeType dateTime = 
                new ExtendedDateTimeType(InfinityType.infinity);
            time.setAfter(dateTime);
            time.setBefore(dateTime);
            this.memberLocator = new LocatorType();
            this.memberLocator.setHandle(
                new HandleType[] { 
                    new HandleType(
                        (String)base.getProperty(ServiceProperties.HANDLE)
                    ) 
                }
            );
            LocatorTypeHolder member = 
                new LocatorTypeHolder(this.memberLocator);
            ExtensibilityType contents = new ExtensibilityType();
            ServiceData data = 
                base.getServiceDataSet().get(ServiceData.CREATE_EXTENSIBILITY);
            if (data == null) {
                data = base.getServiceDataSet().get(ServiceData.INTERFACE);
            }
            if (data != null) {
                contents.set_any(data.externalizeMessage());
            }
            this.registry.add(member, contents, time);
            this.thread = new Thread(
                new ShutdownThread(registry, memberLocator)
            );
            Runtime.getRuntime().addShutdownHook(this.thread);
        } catch (Exception e) {
            logger.error("Failed to publish handle to registry " + 
                         registryHandle, e);
        }
    }

    public QName[] getOperations() {
        return new QName[0];
    }

    public void preCreate(GridServiceBase base) 
        throws GridServiceException {
    }
    public void postCreate(GridContext context) 
        throws GridServiceException {
    }
    public void activate(GridContext context) 
        throws GridServiceException {
    }
    public void deactivate(GridContext context) 
        throws GridServiceException {
    }

    public void preDestroy(GridContext context) 
            throws GridServiceException {
        try {
            this.registry.remove(
                ServiceGroupUtils.getLocatorEquivalenceExpression(
                    this.memberLocator
                )
            );
            Runtime.getRuntime().removeShutdownHook(this.thread);
        } catch (Exception e) {
            logger.error("Failed to unregister handle from registry: " +
                         MessageUtils.toString(e));
        }
    }
}

class ShutdownThread implements Runnable {
    private ServiceGroupRegistration registry;
    private LocatorType locator;
    public ShutdownThread(
        ServiceGroupRegistration registry, 
        LocatorType locator
    ) {
        this.locator = locator;
        this.registry = registry;
    }
    public void run() {
        try {
            this.registry.remove(
                ServiceGroupUtils.getLocatorEquivalenceExpression(
                    this.locator
                )
            );
        } catch (Exception e) {
            RegistryPublishProvider.logger.error(
                "Failed to unregister handle from registry:" + 
                MessageUtils.toString(e)
            );
        }
    }
}
