/*
This file is licensed under the terms of the Globus Toolkit Public
License, found at http://www.globus.org/toolkit/download/license.html.
*/
package org.globus.ogsa.impl.samples.chat;

import org.globus.ogsa.impl.ogsi.GridServiceImpl;
import org.globus.ogsa.impl.security.SecurityManager;
import org.globus.ogsa.impl.security.authentication.Constants;
import org.globus.ogsa.impl.security.authorization.SelfAuthorization;
import org.globus.ogsa.GridContext;
import org.globus.ogsa.GridServiceException;
import org.globus.ogsa.NotificationProvider;
import org.globus.ogsa.ServiceProperties;
import org.globus.ogsa.TopicAlreadyExistsException;
import org.globus.ogsa.samples.chat.MessageData;

import org.globus.axis.gsi.GSIConstants;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import javax.xml.namespace.QName;

import java.rmi.RemoteException;

import java.util.Enumeration;
import java.util.HashMap;
import java.util.Hashtable;
import java.util.Map;


public class ChatImpl extends GridServiceImpl {
    static Log logger = LogFactory.getLog(ChatImpl.class.getName());
    private Map notifyProps;
    private Hashtable whoisTable = new Hashtable();
    private NotificationProvider notificationProvider;

    public ChatImpl() {
        super("Sample Chat Service");
    }

    public void setAlias(String alias) throws RemoteException {
        String userSubject = SecurityManager.getManager().getCaller();

        if ((alias == null) || alias.equals("")) {
            throw new RemoteException("null aliases not allowed");
        }

        if (whoisTable.containsKey(alias)) {
            //Somebody is using this alias; so ignore the request if the
            //current user owns it, or flag it as taken
            String testUserSubject = (String) whoisTable.get(alias);

            if (!testUserSubject.equals(userSubject)) {
                throw new RemoteException("Alias in use by " + testUserSubject);
            }
        } else {
            //Nobody is using this alias, so set or change it as requested
            Enumeration aliases = whoisTable.keys();
            Enumeration userSubjects = whoisTable.elements();
            boolean foundUserSubject = false;

            while (!foundUserSubject && userSubjects.hasMoreElements()) {
                String testUserSubject = (String) userSubjects.nextElement();

                if (testUserSubject.equals(userSubject)) {
                    foundUserSubject = true;

                    String oldAlias = (String) aliases.nextElement();
                    whoisTable.remove(oldAlias);
                } else {
                    aliases.nextElement();
                }
            }

            whoisTable.put(alias, userSubject);
        }
    }

    public String getAlias() throws RemoteException {
        String userSubject = SecurityManager.getManager().getCaller();
        String alias = null;

        //Search the whois table for the alias that maps to the current
        //user's credential subject
        Enumeration aliases = whoisTable.keys();
        Enumeration userSubjects = whoisTable.elements();
        boolean foundUserSubject = false;

        while (!foundUserSubject && userSubjects.hasMoreElements()) {
            String testUserSubject = (String) userSubjects.nextElement();

            if (testUserSubject.equals(userSubject)) {
                foundUserSubject = true;

                alias = (String) aliases.nextElement();
            } else {
                aliases.nextElement();
            }
        }

        if (alias == null) {
            throw new RemoteException(
                "User " + userSubject + " " +
                "does not have an alias registered " +
                "with this service instance"
            );
        }

        return alias;
    }

    public void sendMessage(String message) throws RemoteException {
        String alias = getAlias();

        if (alias != null) {
            MessageData messageData = new MessageData();
            messageData.setText(message);
            messageData.setAlias(alias);
            this.notificationProvider.notify(
                "ChatUpdate", messageData, this.notifyProps
            );
        }
    }

    public String executeCommand(String command) throws RemoteException {
        String commandResult = null;

        String commandName = null;
        int commandDataIndex = command.indexOf(" ");

        if (commandDataIndex == -1) {
            //Commands without data
            throw new RemoteException("Command not recognized: " + commandName);
        } else {
            //Commands with data
            commandName = command.substring(0, commandDataIndex);

            if (commandName.equals("whois")) {
                String commandData = command.substring(commandDataIndex + 1);
                commandResult = whois(commandData);
                logger.debug("Command Data: " + commandData);
            } else {
                throw new RemoteException(
                    "Command not recognized: " + commandName
                );
            }
        }

        return commandResult;
    }

    public void postCreate(GridContext context) throws GridServiceException {
	super.postCreate(context);

        this.notificationProvider =
            (NotificationProvider) getProperty(
                ServiceProperties.NOTIFICATION_SOURCE
            );

	this.notificationProvider.addTopic(
                "ChatUpdate",
                new QName("http://ogsa.globus.org/samples/chat", "messageData")
        );

        SecurityManager manager = SecurityManager.getManager();
        manager.setServiceOwnerFromContext(this);

        this.notifyProps = new HashMap();

        this.notifyProps.put(
            GSIConstants.GSI_MODE, GSIConstants.GSI_MODE_NO_DELEG
        );

        this.notifyProps.put(
            Constants.AUTHORIZATION, SelfAuthorization.getInstance()
        );

        // detect the incoming msg protection and use that for notification
        this.notifyProps.put(
            Constants.GSI_SEC_CONV,
            context.getMessageContext().getProperty(Constants.GSI_SEC_CONV)
        );
    }

    public String whois(String alias) {
        String whoisResult = null;

        whoisResult = (String) whoisTable.get(alias);

        return whoisResult;
    }
}
