package edu.buffalo.cse.jive.app.concurrency;

import java.util.Random;

public abstract class ProducerConsumerFactory {

  public interface Basket {

    public void add(String value);

    public String remove();
  }

  private static final class BasketImpl implements Basket {

    // True if consumer should wait for producer to send message, false
    // if producer should wait for consumer to retrieve message.
    private boolean empty = true;
    // Message sent from producer to consumer.
    private String message;

    @Override
    public synchronized void add(final String message) {

      // Wait until message has been retrieved.
      while (!empty) {
        try {
          wait();
        }
        catch (final InterruptedException e) {
        }
      }
      // Toggle status.
      empty = false;
      // Store message.
      this.message = message;
      // Notify consumer that status has changed.
      notifyAll();
    }

    @Override
    public synchronized String remove() {

      // Wait until message is available.
      while (empty) {
        try {
          wait();
        }
        catch (final InterruptedException e) {
        }
      }
      // Toggle status.
      empty = true;
      // Notify producer that status has changed.
      notifyAll();
      return message;
    }
  }

  public interface Consumer extends Runnable {

  }

  private static final class ConsumerImpl implements Consumer {

    private final Basket basket;

    private ConsumerImpl(final Basket basket) {

      this.basket = basket;
    }

    @Override
    public void run() {

      final Random random = new Random();
      for (String message = basket.remove(); !message.equals("DONE"); message = basket.remove()) {
        // System.out.format("Consuming message: %s%n", message);
        try {
          Thread.sleep(random.nextInt(ITERATION_SLEEP));
        }
        catch (final InterruptedException e) {
        }
      }
      // System.out.println("Termination signal: DONE");
    }
  }

  public interface Producer extends Runnable {

  }

  private static final class ProducerImpl implements Producer {

    private final Basket basket;
    private final String[] messages;

    private ProducerImpl(final Basket basket) {

      this(basket, DEFAULT_MESSAGES);
    }

    private ProducerImpl(final Basket basket, final String[] messages) {

      this.basket = basket;
      this.messages = messages;
    }

    @Override
    public void run() {

      final Random random = new Random();

      for (final String message : messages) {
        basket.add(message);
        try {
          Thread.sleep(random.nextInt(ITERATION_SLEEP));
        }
        catch (final InterruptedException e) {
        }
      }
      basket.add("DONE");
    }
  }

  private static final String DEFAULT_MESSAGES[] = { "Mares eat oats", "Dogs eat oats",
      "Little lambs eat ivy", "A kid will eat ivy too" };
  private static final int ITERATION_SLEEP = 50;

  public static Basket createBasket() {

    return new BasketImpl();
  }

  public static Consumer createConsumer(final Basket basket) {

    return new ConsumerImpl(basket);
  }

  public static Producer createProducer(final Basket basket) {

    return new ProducerImpl(basket);
  }

  public static Producer createProducer(final Basket basket, final String[] messages) {

    return new ProducerImpl(basket, messages);
  }
}
