package edu.buffalo.cse.jive.app.sorting;

import java.util.concurrent.CountDownLatch;

public abstract class SortAlgorithmFactory {

  abstract <E extends Comparable<E>> Thread createSorterThread(SortAlgorithm<E> algorithm);

  protected abstract <E extends Comparable<E>> Thread newThread(final SortAlgorithm<E> algorithm);

  public static <E extends Comparable<E>> SortAlgorithm<E> createAlgorithm(
      final AlgorithmId algoId, final SortableCollection<E> values, final boolean isAnimated) {

    return createAlgorithm(algoId, values, null, isAnimated);
  }

  public static <E extends Comparable<E>> SortAlgorithm<E> createAlgorithm(
      final AlgorithmId algoId, final SortableCollection<E> values,
      final CountDownLatch startSignal, final CountDownLatch endSignal, final boolean isAnimated) {

    return createAlgorithm(algoId, values, new SynchronizedFactory(startSignal, endSignal),
        isAnimated);
  }

  private static <E extends Comparable<E>> SortAlgorithm<E> createAlgorithm(
      final AlgorithmId algoId, final SortableCollection<E> values,
      final SortAlgorithmFactory factory, final boolean isAnimated) {

    final SortAlgorithmFactory actualFactory = (factory == null ? new DefaultFactory() : factory);
    if (algoId != null) {
      return algoId.createAlgorithm(actualFactory, values, isAnimated);
    }
    throw new IllegalArgumentException("Cannot create a null sort algorithm.");
  }

  private static class DefaultFactory extends SortAlgorithmFactory {

    @Override
    final <E extends Comparable<E>> Thread createSorterThread(final SortAlgorithm<E> algorithm) {

      final Thread sorter = newThread(algorithm);
      sorter.setDaemon(true);
      return sorter;
    }

    @Override
    protected <E extends Comparable<E>> Thread newThread(final SortAlgorithm<E> algorithm) {

      return new Thread(new Runnable() {

        @Override
        public void run() {

          algorithm.sort();
          algorithm.done();
        }

      }, algorithm.getId().algoName());
    }
  }

  private static class SynchronizedFactory extends DefaultFactory {

    private final CountDownLatch endSignal;
    private final CountDownLatch startSignal;

    public SynchronizedFactory(final CountDownLatch startSignal, final CountDownLatch endSignal) {

      this.startSignal = startSignal;
      this.endSignal = endSignal;
    }

    @Override
    protected <E extends Comparable<E>> Thread newThread(final SortAlgorithm<E> algorithm) {

      return new Thread(new Sorter(startSignal, endSignal, algorithm), algorithm.getId().algoName());
    }
  }
}
