package edu.buffalo.cse.jive.app.sorting.animation;

import java.awt.Dimension;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.lang.reflect.InvocationTargetException;

import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.SwingConstants;
import javax.swing.SwingUtilities;

import edu.buffalo.cse.jive.app.sorting.AlgorithmId;
import edu.buffalo.cse.jive.app.sorting.SortAlgorithm.TerminationListener;
import edu.buffalo.cse.jive.app.sorting.SortApplication;

public class SortAnimation implements TerminationListener, ActionListener {

  private final boolean autoMode;
  private final JPanel bubble = new JPanel();
  private final JPanel control = new JPanel();
  private final int dataSize;
  protected boolean done = false;
  private final JFrame frame = new JFrame("Animation of Common Sort Algorithms");
  private final JPanel insertion = new JPanel();
  private final JLabel label = new JLabel("Click to (re)start the animation");
  private final JPanel merge = new JPanel();
  private final JPanel quick = new JPanel();
  private final JPanel selection = new JPanel();
  private SortApplication sorter;
  private final JButton startButton = new JButton("Start!");

  public SortAnimation(final int dataSize) {

    this(dataSize, false);
  }

  public SortAnimation(final int dataSize, final boolean autoMode) {

    this.autoMode = autoMode;
    this.dataSize = dataSize;
  }

  @Override
  public void actionPerformed(final ActionEvent e) {

    if ("start".equals(e.getActionCommand()) && startButton.isEnabled()) {
      start();
    }
  }

  @Override
  public synchronized void done(final AlgorithmId algorithm) {

    for (final AlgorithmId algoId : AlgorithmId.values()) {
      if (!sorter.getAlgorithm(algoId).isDone()) {
        return;
      }
    }
    SwingUtilities.invokeLater(new Runnable() {

      @Override
      public void run() {

        if (autoMode) {
          try {
            Thread.sleep(500);
          }
          catch (final InterruptedException e) {
            e.printStackTrace();
          }
          frame.dispose();
          done = true;
        }
        else {
          startButton.setEnabled(true);
        }
      }
    });
  }

  public boolean isDone() {

    return done;
  }

  private void prepareSorters() {

    if (sorter != null) {
      bubble.removeAll();
      insertion.removeAll();
      merge.removeAll();
      selection.removeAll();
      quick.removeAll();
    }
    this.sorter = new SortApplication(dataSize, true);
    bubble.add(new SortPanel(sorter.getAlgorithm(AlgorithmId.BUBBLE_SORT)));
    insertion.add(new SortPanel(sorter.getAlgorithm(AlgorithmId.INSERTION_SORT)));
    merge.add(new SortPanel(sorter.getAlgorithm(AlgorithmId.MERGE_SORT)));
    selection.add(new SortPanel(sorter.getAlgorithm(AlgorithmId.SELECTION_SORT)));
    quick.add(new SortPanel(sorter.getAlgorithm(AlgorithmId.QUICK_SORT)));
    for (final AlgorithmId algoId : AlgorithmId.values()) {
      sorter.getAlgorithm(algoId).addListener(this);
    }
  }

  public void show() {

    try {
      SwingUtilities.invokeAndWait(new Runnable() {

        @Override
        public void run() {

          startButton.setActionCommand("start");
          startButton.setPreferredSize(new Dimension(120, 30));
          startButton.addActionListener(SortAnimation.this);
          label.setHorizontalAlignment(SwingConstants.CENTER);
          label.setPreferredSize(new Dimension(240, 90));
          control.add(label);
          control.add(startButton);
          frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
          frame.setLayout(new GridLayout(2, 3, 5, 5));
          frame.add(control);
          frame.add(bubble);
          frame.add(insertion);
          frame.add(merge);
          frame.add(selection);
          frame.add(quick);
          frame.setSize(800, 600);
          frame.setVisible(true);
        }
      });
    }
    catch (final InterruptedException e) {
      e.printStackTrace();
    }
    catch (final InvocationTargetException e) {
      e.printStackTrace();
    }
    if (autoMode) {
      start();
    }
  }

  private void start() {

    if (autoMode) {
      try {
        Thread.sleep(100);
      }
      catch (final InterruptedException e) {
        e.printStackTrace();
      }
    }
    SwingUtilities.invokeLater(new Runnable() {

      @Override
      public void run() {

        startButton.setEnabled(false);
        prepareSorters();
        new Thread(new Runnable() {

          @Override
          public void run() {

            sorter.start();
          }
        }).start();
      }
    });
  }
}