package edu.buffalo.cse.jive.app.dblp.sql;

import java.io.IOException;
import java.io.Writer;
import java.util.Map;

import edu.buffalo.cse.jive.app.dblp.model.Entity;
import edu.buffalo.cse.jive.app.dblp.model.Persister;
import edu.buffalo.cse.jive.app.dblp.model.Entity.TypeEntity;

public class SQLPersister implements Persister {

  private static final String FN_AUTHORS = "insert-author.sql";
  private static final String FN_EDITORS = "insert-editor.sql";

  private static final String FN_ENTITY = "insert-entity.sql";
  private static final String FN_ENTITY_AUTHOR = "insert-entity-author.sql";
  private static final String FN_ENTITY_CDROM = "insert-entity-cdrom.sql";
  private static final String[] FN_ENTITY_CHILD = { "insert-entity-article.sql",
      "insert-entity-book.sql", "insert-entity-incollection.sql",
      "insert-entity-inproceedings.sql", "insert-entity-mastersthesis.sql",
      "insert-entity-phdthesis.sql", "insert-entity-proceedings.sql", "insert-entity-www.sql" };
  private static final String FN_ENTITY_CITE = "insert-entity-cite.sql";
  private static final String FN_ENTITY_EDITOR = "insert-entity-editor.sql";
  private static final String FN_ENTITY_EE = "insert-entity-ee.sql";
  private static final String FN_ENTITY_ISBN = "insert-entity-isbn.sql";
  private static final String FN_ENTITY_NOTE = "insert-entity-note.sql";
  private static final String FN_ENTITY_PUBLISHER = "insert-entity-publisher.sql";
  private static final String FN_ENTITY_URL = "insert-entity-url.sql";

  private static final String FN_PUBLISHERS = "insert-publisher.sql";
  public static String TX_COMMIT = "COMMIT;\n";
  public static String TX_START = ""; // "START TRANSACTION;\n";

  private int entityCounter = 0;
  private final SQLWriterFactory factory;
  private final String folderName;
  private SQLStatements generator;
  private final Writer[] handleChild = new Writer[TypeEntity.values().length];
  private final Writer handleEntity;
  private final Writer handleEntityAuthor;
  private final Writer handleEntityCdrom;
  private final Writer handleEntityCite;
  private final Writer handleEntityEditor;
  private final Writer handleEntityEe;
  private final Writer handleEntityIsbn;
  private final Writer handleEntityNote;
  private final Writer handleEntityPublisher;
  private final Writer handleEntityUrl;
  private final boolean protectInTransaction;

  // private final ByteBuffer buffer = ByteBuffer.allocate(4096);

  public SQLPersister(final String folderName, final SQLWriterFactory factory,
      final boolean protectInTransaction) throws IOException {

    this.protectInTransaction = protectInTransaction;
    this.folderName = folderName;
    this.factory = factory;
    handleEntity = factory.newWriter(folderName + FN_ENTITY);
    for (final TypeEntity e : TypeEntity.values()) {
      if (e != TypeEntity.www) {
        handleChild[e.ordinal()] = factory.newWriter(folderName + FN_ENTITY_CHILD[e.ordinal()]);
      }
    }
    handleEntityAuthor = factory.newWriter(folderName + FN_ENTITY_AUTHOR);
    handleEntityEditor = factory.newWriter(folderName + FN_ENTITY_EDITOR);
    handleEntityPublisher = factory.newWriter(folderName + FN_ENTITY_PUBLISHER);
    handleEntityCdrom = factory.newWriter(folderName + FN_ENTITY_CDROM);
    handleEntityCite = factory.newWriter(folderName + FN_ENTITY_CITE);
    handleEntityEe = factory.newWriter(folderName + FN_ENTITY_EE);
    handleEntityIsbn = factory.newWriter(folderName + FN_ENTITY_ISBN);
    handleEntityNote = factory.newWriter(folderName + FN_ENTITY_NOTE);
    handleEntityUrl = factory.newWriter(folderName + FN_ENTITY_URL);
    if (protectInTransaction) {
      writeToEntities(TX_START);
    }
  }

  @Override
  public int entityCounter() {

    return entityCounter;
  }

  @Override
  public void prepare() throws IOException {

    // no-op
  }

  @Override
  public void flush() {

    try {
      // persistEntities();
      if (protectInTransaction) {
        writeToEntities(TX_COMMIT);
      }
      flushForEntity();
      persistAuthors();
      persistEditors();
      persistPublishers();
    }
    catch (final IOException e) {
      // TODO Auto-generated catch block
      e.printStackTrace();
    }
  }

  private void flushForEntity() {

    try {
      handleEntity.close();
      for (final TypeEntity e : TypeEntity.values()) {
        if (e != TypeEntity.www) {
          handleChild[e.ordinal()].close();
        }
      }
      handleEntityAuthor.close();
      handleEntityEditor.close();
      handleEntityPublisher.close();
      handleEntityCdrom.close();
      handleEntityCite.close();
      handleEntityEe.close();
      handleEntityIsbn.close();
      handleEntityNote.close();
      handleEntityUrl.close();
    }
    catch (final IOException e) {
      // TODO Auto-generated catch block
      e.printStackTrace();
    }
  }

  private SQLStatements getRecords() {

    if (generator == null) {
      generator = new SQLStatements();
    }
    return generator;
  }

  @Override
  public void persist(final Object source) throws IOException {

    final Entity entity = (Entity) source;
    // generic entity record
    write(handleEntity, getRecords().getEntity(entity));
    // specific entity record
    if (entity.getType() != TypeEntity.www) {
      write(handleChild[entity.getType().ordinal()], getRecords().getChild(entity));
    }
    writeEntityAuthor(entity);
    writeEntityEditor(entity);
    writeEntityPublisher(entity);
    writeEntityCdroms(entity);
    writeEntityCites(entity);
    writeEntityEes(entity);
    writeEntityIsbns(entity);
    writeEntityNotes(entity);
    writeEntityUrls(entity);
    entityCounter++;
  }

  private void persistAuthors() {

    final Map<String, Integer> authors = Entity.getLookupAuthors();
    System.out.println("author count: " + authors.size());
    try {
      final Writer out = factory.newWriter(folderName + FN_AUTHORS);
      if (protectInTransaction) {
        write(out, TX_START);
      }
      for (final String author : authors.keySet()) {
        write(out, getRecords().getLookupAuthor(authors.get(author), author));
      }
      if (protectInTransaction) {
        write(out, TX_COMMIT);
      }
      out.close();
    }
    catch (final IOException e) {
      e.printStackTrace();
    }
  }

  private void persistEditors() {

    final Map<String, Integer> editors = Entity.getLookupEditors();
    System.out.println("editor count: " + editors.size());
    try {
      final Writer out = factory.newWriter(folderName + FN_EDITORS);
      if (protectInTransaction) {
        write(out, TX_START);
      }
      for (final String editor : editors.keySet()) {
        write(out, getRecords().getLookupEditor(editors.get(editor), editor));
      }
      if (protectInTransaction) {
        write(out, TX_COMMIT);
      }
      out.close();
    }
    catch (final IOException e) {
      e.printStackTrace();
    }
  }

  private void persistPublishers() {

    final Map<String, Integer> publishers = Entity.getLookupPublishers();
    System.out.println("publisher count: " + publishers.size());
    try {
      final Writer out = factory.newWriter(folderName + FN_PUBLISHERS);
      if (protectInTransaction) {
        write(out, TX_START);
      }
      for (final String publisher : publishers.keySet()) {
        write(out, getRecords().getLookupPublisher(publishers.get(publisher), publisher));
      }
      if (protectInTransaction) {
        write(out, TX_COMMIT);
      }
      out.close();
    }
    catch (final IOException e) {
      // TODO Auto-generated catch block
      e.printStackTrace();
    }
  }

  // private void cycleFile(Writer handle) {
  //
  // fileCount++;
  // flush();
  // try {
  // outEntities = new OutputStreamWriter(new BufferedOutputStream(new FileOutputStream(folderName
  // + FN_ENTITIES + fileCount, true)));
  // }
  // catch (FileNotFoundException e) {
  // // TODO Auto-generated catch block
  // e.printStackTrace();
  // }
  // }

  private void rewrite(final Writer out, final String value) throws IOException {

    // buffer.rewind();
    out.write(value);
    // out.force(true);
  }

  private void write(final Writer out, final String value) throws IOException {

    // buffer.clear();
    // buffer.asCharBuffer().put(value);
    // buffer.flip();
    out.write(value);
    // out.force(true);
  }

  private void writeEntityAuthor(final Entity entity) throws IOException {

    for (final Integer authorId : entity.getAuthors()) {
      write(handleEntityAuthor, getRecords().getEntityAuthor(entity, authorId));
    }
  }

  private void writeEntityCdroms(final Entity entity) throws IOException {

    for (final String cdrom : entity.getCdroms()) {
      write(handleEntityCdrom, getRecords().getEntityCdrom(entity, cdrom));
    }
  }

  private void writeEntityCites(final Entity entity) throws IOException {

    for (final String cite : entity.getCites()) {
      write(handleEntityCite, getRecords().getEntityCite(entity, cite));
    }
  }

  private void writeEntityEditor(final Entity entity) throws IOException {

    for (final Integer editorId : entity.getEditors()) {
      write(handleEntityEditor, getRecords().getEntityEditor(entity, editorId));
    }
  }

  private void writeEntityEes(final Entity entity) throws IOException {

    for (final String ee : entity.getEes()) {
      write(handleEntityEe, getRecords().getEntityEe(entity, ee));
    }
  }

  private void writeEntityIsbns(final Entity entity) throws IOException {

    for (final String isbn : entity.getIsbns()) {
      write(handleEntityIsbn, getRecords().getEntityIsbn(entity, isbn));
    }
  }

  private void writeEntityNotes(final Entity entity) throws IOException {

    for (final String note : entity.getNotes()) {
      write(handleEntityNote, getRecords().getEntityNote(entity, note));
    }
  }

  private void writeEntityPublisher(final Entity entity) throws IOException {

    for (final Integer publisherId : entity.getPublishers()) {
      write(handleEntityPublisher, getRecords().getEntityPublisher(entity, publisherId));
    }
  }

  private void writeEntityUrls(final Entity entity) throws IOException {

    for (final String url : entity.getUrls()) {
      write(handleEntityUrl, getRecords().getEntityUrl(entity, url));
    }
  }

  private void writeToEntities(final String value) throws IOException {

    write(handleEntity, value);
    for (final TypeEntity e : TypeEntity.values()) {
      if (e != TypeEntity.www) {
        rewrite(handleChild[e.ordinal()], value);
      }
    }
    rewrite(handleEntityAuthor, value);
    rewrite(handleEntityEditor, value);
    rewrite(handleEntityPublisher, value);
    rewrite(handleEntityCdrom, value);
    rewrite(handleEntityCite, value);
    rewrite(handleEntityEe, value);
    rewrite(handleEntityIsbn, value);
    rewrite(handleEntityNote, value);
    rewrite(handleEntityUrl, value);
  }
}