package edu.buffalo.cse.jive.benchmarks;

import java.net.URI;
import java.net.URISyntaxException;

public abstract class Benchmark {

	private long startTime;
	private long elapsedTime;

	protected URI getResource(final String resourcePath)
			throws BenchmarkException {

		try {
			return Benchmark.class.getResource(resourcePath).toURI();
		} catch (final URISyntaxException urise) {
			throw new BenchmarkException(urise);
		}
	}

	protected static void message(final String message) {

		System.err.println(message);
	}

	protected static void message(final String message, Object... args) {

		System.err.println(String.format(message, args));
	}

	protected abstract String name();

	protected abstract void doRun() throws BenchmarkException;

	public void run() {

		try {
			doRun();
		} catch (final Exception e) {
			message("Exception while running benchmark.");
			e.printStackTrace();
		}
	}

	public static class BenchmarkException extends Exception {

		private static final long serialVersionUID = -8122089456764966900L;

		public BenchmarkException(final String message) {

			super(message);
		}

		public BenchmarkException(final Throwable cause) {

			super(cause);
		}
	}

	private static double mean(final long[] values) {

		double result = 0;
		for (final long n : values) {
			result += n;
		}
		return 1.0 * result / values.length;
	}

	private static double stdev(final double mean, final long[] values) {

		double result = 0;
		for (final long n : values) {
			result += (mean - n) * (mean - n);
		}
		return StrictMath.sqrt(result / (values.length - 1));
	}

	public static void run(Class<? extends Benchmark> benchmark, int runCount)
			throws BenchmarkException {

		try {
			// vm warm-up
			if (runCount > 1) {
				final Benchmark warmup = ((Benchmark) benchmark.newInstance());
				message("Warming up JVM for benchmark '%s'.", warmup.name());
				warmup.run();
			}
			// actual benchmark
			final long[] runs = new long[runCount];
			for (int i = 0; i < runCount; i++) {
				final Benchmark b = ((Benchmark) benchmark.newInstance());
				message("Benchmark '%s' run #%d.", b.name(), i + 1);
				b.timeStart();
				b.run();
				b.timeStop();
				message("%8.4f sec.", 1.0 * b.elapsedTime
						/ (1000 * 1000 * 1000));
				runs[i] = b.elapsedTime;
			}
			if (runs.length > 1) {
				final double mean = mean(runs);
				final double stdev = stdev(mean, runs);
				message("AVG: %8.4f sec, STDEV: %8.4f sec", mean
						/ (1000 * 1000 * 1000), stdev / (1000 * 1000 * 1000));
			}
		} catch (final InstantiationException ie) {
			throw new BenchmarkException(ie);
		} catch (final IllegalAccessException iae) {
			throw new BenchmarkException(iae);
		}
	}

	protected void timeStop() {

		elapsedTime = System.nanoTime() - startTime;
	}

	protected void timeStart() {

		startTime = System.nanoTime();
	}
}