package edu.buffalo.cse.jive.data;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Random;
import java.util.Set;

public class DataFactory {

  /**
   * Creates a list of random double values, uniformly distributed between 0 and 1.
   */
  public static final List<Double> createDoubleList(final int dataSize) {

    final List<Double> data = new ArrayList<Double>();
    final Random rand = new java.util.Random();
    for (int i = 0; i < dataSize; i++) {
      data.add(rand.nextDouble());
    }
    return data;
  }

  /**
   * Creates a list of random double values, uniformly distributed in the given range.
   */
  public static final List<Double> createDoubleList(final int dataSize, final int min, final int max) {

    final List<Double> data = new ArrayList<Double>();
    for (final Double d : createDoubleList(dataSize)) {
      data.add(min + (max - min) * d);
    }
    return data;
  }

  /**
   * Creates a set of random double values, uniformly distributed between 0 and 1.
   */
  public static final Set<Double> createDoubleSet(final int dataSize) {

    final Set<Double> data = new HashSet<Double>();
    final Random rand = new java.util.Random();
    while (dataSize > data.size()) {
      data.add(rand.nextDouble());
    }
    return data;
  }

  /**
   * Creates a set of random double values, uniformly distributed in the given range.
   */
  public static final Set<Double> createDoubleSet(final int dataSize, final int min, final int max) {

    final Set<Double> data = new HashSet<Double>();
    for (final Double d : createDoubleSet(dataSize)) {
      data.add(min + (max - min) * d);
    }
    return data;
  }

  /**
   * Creates a list of random integer values, uniformly distributed in the valid integer range.
   */
  public static final List<Integer> createIntegerList(final int dataSize) {

    final List<Integer> data = new ArrayList<Integer>();
    final Random rand = new java.util.Random();
    for (int i = 0; i < dataSize; i++) {
      data.add(rand.nextInt());
    }
    return data;
  }

  /**
   * Creates a list of random integer values, uniformly distributed in the given range.
   */
  public static final List<Integer> createIntegerList(final int dataSize, final int min,
      final int max) {

    final List<Integer> data = new ArrayList<Integer>();
    final Random rand = new java.util.Random();
    for (int i = 0; i < dataSize; i++) {
      data.add(min + rand.nextInt((max - min)));
    }
    return data;
  }

  /**
   * Creates a set of random integer values, uniformly distributed in the given range.
   */
  public static final Set<Integer> createIntegerSet(final int dataSize) {

    final Set<Integer> data = new HashSet<Integer>();
    final Random rand = new java.util.Random();
    while (dataSize > data.size()) {
      data.add(rand.nextInt());
    }
    return data;
  }

  /**
   * Creates a set of random integer values, uniformly distributed in the given range.
   */
  public static final Set<Integer> createIntegerSet(final int dataSize, final int min, final int max) {

    final Set<Integer> data = new HashSet<Integer>();
    final Random rand = new java.util.Random();
    while (dataSize > data.size()) {
      data.add(min + rand.nextInt((max - min)));
    }
    return data;
  }

  /**
   * Loads a list of double values from the file with the given name. The file should contain one
   * value per line. Any line that cannot be parsed as a double value is ignored.
   */
  public static List<Double> loadAsDoubleList(final String fileName) throws IOException {

    final List<Double> result = new LinkedList<Double>();
    final BufferedReader reader = new BufferedReader(new FileReader(new File(fileName)));
    String line = null;
    try {
      while ((line = reader.readLine()) != null) {
        try {
          result.add(Double.parseDouble(line));
        }
        catch (final NumberFormatException nfe) {
          // cowardly skip this line
        }
      }
    }
    finally {
      reader.close();
    }
    return result;
  }

  /**
   * Loads a list of lines from the file with the given name.
   */
  public static List<String> loadAsStringList(final String fileName) throws IOException {

    final List<String> result = new LinkedList<String>();
    final BufferedReader reader = new BufferedReader(new FileReader(new File(fileName)));
    String line = null;
    try {
      while ((line = reader.readLine()) != null) {
        result.add(line);
      }
    }
    finally {
      reader.close();
    }
    return result;
  }

  /**
   * Loads a list of numbers as integer values from the file with the given name. The list is first
   * loaded as a list of doubles and then converted to a list of integers. The side-effect of this
   * approach is that double values are be converted to appropriate integer values.
   */
  public static List<Integer> loadAsIntegerList(final String fileName) throws IOException {

    final List<Double> temp = loadAsDoubleList(fileName);
    final List<Integer> result = new LinkedList<Integer>();
    for (final Double d : temp) {
      result.add(d.intValue());
    }
    return result;
  }

  /**
   * Creates a list of random double values, normally distributed with mean 0 and standard deviation
   * 1.
   */
  public static final List<Double> normalDoubleList(final int dataSize) {

    final List<Double> data = new ArrayList<Double>();
    final Random rand = new java.util.Random();
    for (int i = 0; i < dataSize; i++) {
      data.add(rand.nextGaussian());
    }
    return data;
  }

  /**
   * Creates a list of random double values, normally distributed in the given range.
   */
  public static final List<Double> normalDoubleList(final int dataSize, final double mean,
      final double stdev) {

    final List<Double> data = new ArrayList<Double>();
    for (final Double d : normalDoubleList(dataSize)) {
      data.add(mean + stdev * d);
    }
    return data;
  }

  /**
   * Creates a list of random integer values, uniformly distributed in the given range.
   */
  public static final List<Integer> normalIntegerList(final int dataSize, final int mean,
      final int stdev) {

    final List<Integer> data = new ArrayList<Integer>();
    for (final Double d : normalDoubleList(dataSize, mean, stdev)) {
      data.add(d.intValue());
    }
    return data;
  }

  /**
   * Saves the list of numbers as double values to the file with the given name. The file contains
   * one value per line.
   */
  public static void saveAsDoubleList(final Collection<? extends Number> list, final String fileName)
      throws IOException {

    final BufferedWriter writer = new BufferedWriter(new FileWriter(new File(fileName)));
    try {
      for (final Number n : list) {
        writer.write(Double.toString(n.doubleValue()));
        writer.write("\n");
      }
    }
    finally {
      writer.close();
    }
  }

  /**
   * Saves the list of numbers as integer values to the file with the given name. The file contains
   * one value per line.
   */
  public static void saveAsIntegerList(final Collection<? extends Number> list,
      final String fileName) throws IOException {

    final BufferedWriter writer = new BufferedWriter(new FileWriter(new File(fileName)));
    try {
      for (final Number n : list) {
        writer.write(String.valueOf(n.intValue()));
        writer.write("\n");
      }
    }
    finally {
      writer.close();
    }
  }
}