// ============================================================================
// main.cpp
// ~~~~~~~~
// author: hqn
// - main body of the txtparser program
// ============================================================================

#include <iostream>
#include <fstream>
#include <stdexcept>
#include <string>   
#include <map>   
#include <cstdlib> // for exit()

#include "term_control.h"
#include "error_handling.h"
#include "Lexer.h"

using namespace std;

typedef void (*cmd_handler_t)(Lexer);

void numcols(Lexer);
void numrows(Lexer);
void bye(Lexer);

void prompt();

const char *usage_msg = "Usage: numcols <file name>\n"
                        "       numrows <file name>\n"
                        "       exit/quit/bye";

int main(int argc, char **argv) 
{
    if (argc != 1) 
        error_quit("txtparser doesn't take any argument");

    map<string, cmd_handler_t> commands;
    commands["numcols"] = &numcols;
    commands["numrows"] = &numrows;
    commands["quit"] = &bye;
    commands["exit"] = &bye;
    commands["bye"] = &bye;

    string line; Token tok; Lexer lexer;

    while (cin) {
        prompt(); 
        getline(cin, line); 
        lexer.set_input(line);

        if (!lexer.has_more_token()) continue;

        tok = lexer.next_token();
        if (tok.type == IDENT) {
            if (commands.find(tok.value) != commands.end()) {
                try {
                    commands[tok.value](lexer);
                } catch (exception &e) {
                    error_return(e.what());
                }
                continue;
            }
        }
        note(usage_msg);
    }

    return 0; // 0 indicates 'success', this is a Unix thing
}

void numcols(Lexer lex)
{
    Token tok;
    if (!lex.has_more_token())
        throw runtime_error(usage_msg);

    tok = lex.next_token();
    if (lex.has_more_token()) // 3rd token
        throw runtime_error(usage_msg);

    ifstream ifs; // input file stream                                          
    ifs.open(tok.value.c_str());                                                
    if (ifs.fail()) {                                                       
        string msg("Failed to open file ");
        msg += tok.value;
        ifs.clear();                                                        
        throw runtime_error(msg);
    } else {                                                                
        size_t width = 0;
        string line;
        while (getline(ifs, line)) {
            Lexer my_lexer(line);
            size_t local_width = 0;
            while (my_lexer.next_token().type != ENDTOK)
                local_width++;
            width = max(width, local_width);
        }
        cout << term_cc(YELLOW) << width << term_cc() << endl;
    }                     
    ifs.close(); // always remember to close it                         
}

void numrows(Lexer lex)
{
    Token tok;
    if (!lex.has_more_token())
        throw runtime_error(usage_msg);

    tok = lex.next_token();
    if (lex.has_more_token())
        throw runtime_error(usage_msg);

    ifstream ifs; // input file stream                                          
    ifs.open(tok.value.c_str());                                                
    if (ifs.fail()) {                                                       
        string msg("ERROR: Failed to open file ");
        msg += tok.value;
        ifs.clear();                                                        
        throw runtime_error(msg);
    } else {                                                                
        size_t count = 0;
        string line;
        while (getline(ifs, line))                                          
            count++;
        cout << term_cc(YELLOW) << count << term_cc() << endl;
    }                     
    ifs.close(); // always remember to close it                         
}

void bye(Lexer l) 
{
    if (l.has_more_token())
        throw runtime_error(usage_msg);
    else 
        exit(0);
}

void prompt()
{ 
    cout << term_cc(BLUE) << "> " << term_cc() << flush; 
}
