// ============================================================================
// BinaryTree.h
// ~~~~~~~~~~~~
// hqn
// - A Very Basic Binary Tree Class
// - Nodes have string payloads
// ============================================================================

#ifndef BINARY_TREE_H_
#define BINARY_TREE_H_

#include <vector>
#include <string>
#include <map>

class BinaryTree
{
private:
    struct Node 
    {
        std::string payload;
        Node* left;
        Node* right;
        Node(std::string s = std::string(),
             Node* l = NULL, Node* r = NULL) : payload(s), left(l), right(r) 
        {}
    };

    struct Text 
    {
        std::string text; // the text to be printed, e. "abc__"
        size_t pos;       // its position on the line
        Text(std::string t="", size_t p=0) : text(t), pos(p) {}
    };

    struct Coordinate 
    {
        size_t width;  // width of a subtree box
        size_t offset; // offset to where the parent points
    };

public:
    // default constructor
    BinaryTree();

    // this constructor constructs a tree from a preorder and an inorder vector
    BinaryTree(std::vector<std::string>& a, std::vector<std::string>& b);

    // destructor
    ~BinaryTree();

    /**
     * this changes the current tree to a new tree, using the 
     * preorder sequence a and inorder sequence b
     * - return true if the input sequences are good
     * - return false if they don't represent a tree
     */
    bool setTree(std::vector<std::string>& a, std::vector<std::string>& b);

    // simple size check
    bool isEmpty() const;

    // common traversal sequences
    std::string levelOrderSequence();
    std::string inOrderSequence();
    std::string postOrderSequence();
    std::string preOrderSequence();

    /**
     * -----------------------------------------------------------------------------
     * print a tree horizontally; the algorithm goes like this:
     * - for each node in the tree we compute the "width" of its subtree
     * - then do a level order traversal of the tree
     * - we keep 2 counter variables c & n
     *   c counts # of nodes in the current level still in the Q
     *   n counts # of nodes in the next level already pushed in the Q
     *   when c reaches 0, we know the current level is done and we move on to the
     *   next level; 
     * - Before moving on to the next level, we print the current level line 
     *   which was built up while extracting nodes from the current level out 
     *   from the Q
     * - Now, to build the "current level line" to be printed, consider an example
     *   where nodes at the current level are
     *   abc     xyz     uv  def
     * - for each node, we compute the "indentation" for that node. The 
     *   indentation is computed using width_ which we compute using 
     *   the computeSubtreeWidths() helper function. 
     *      width_[node] is the width of the subtree rooted at node
     * - the parent node "tells" its child what the indentation of the child is
     *     + the left child has indentation exactly equal to the parent
     *     + the right child has indentation equal to 
     *       the parent's indentation + max(left-width, payload) + 1 
     *
     *  payload___
     *  |         \  <--- these are called "connectives", stored in conn_vec
     *  ---------- ----------
     *  | left   | | right  |
     *  | width  | | width  |
     *  ---------- ----------
     * - once we know the indentation of each node, building up the line to be
     *   printed is not hard
     * -----------------------------------------------------------------------------
     */
    std::string horizontal();

    /**
     * ------------------------------------------------------------------------
     * To print the tree vertically, call verticalHelper(node, path) which is 
     * recursive. The algorithm works as follows. 
     * Consider the following tree as an example
     *
     * root
     * |__34
     * |  |__c
     * |  |                 <--- transitional line
     * |  |__longer
     * |     |__bbbbbbb
     * |     |              <--- transitional line
     * |     |__aaaaaaaaa
     * |                    <--- transitional line
     * |__thisisverylong
     *    |__short
     *    |                 <--- transitional line
     *    |__x
     *
     * we are given a root of a subtree, say with payload "34" above. 
     * The "path" variable indicates the branches we took until this node, 
     * Given the path, it is easy to decide where to print the bars, using
     * printBars(), then we print the node's payload
     *
     * Finally, we
     * - recursively print the right subtree
     * - print the transitional line between two children
     * - recursively print the left subtree
     * ------------------------------------------------------------------------
     */
    std::string vertical();

    /**
     * ------------------------------------------------------------------------
     * print a tree symmetrically; the algorithm is very much similar to the
     * algorithm for horizontal print -- except that we need to be more careful
     * on how to compute where to print the nodes and the ___NODE___ lines
     * around each node.
     * ------------------------------------------------------------------------
     */
    std::string symmetric();

private:
    /** 
     * auxiliary routine called by setTree to construct a new tree.
     * recursively construct a tree from a preorder vector and an inorder vector
     * a[a_start, ..., a_end] is the preorder sequence
     * b[b_start, ..., b_end] is the inorder sequence
     * where a_end = a_start + len
     *       b_end = b_start + len
     * this routines assumes that the preorder sequence is a permutation of
     * the inorder sequence
     */
    Node* constructTree(std::vector<std::string>& a, size_t a_start, 
                        std::vector<std::string>& b, size_t b_start, 
                        size_t len);

    // check if a vector of strings is a permutation of another
    bool isPermutation(std::vector<std::string> a,
                       std::vector<std::string> b);

    // helper for inorder sequence
    std::string inOrderSequence(Node*);
    std::string postOrderSequence(Node*);
    std::string preOrderSequence(Node*);

    // helper function for destructor
    void clearTree(Node*);

    /**
     * ** helper functions for vertical print **
     *
     * path is a boolean vector indicating the path from the root to
     * the current node; for example, [ true, false, true ] means the
     * path is [ right, left, right ]
     * path is passed by reference for efficiency purposes, but this requires
     * the caller to do pop_back correctly
     * "underscore" indicates whether we should print |__ part or not
     */
    std::string verticalHelper(Node* node, std::vector<bool>& path);
    std::string printBars(std::vector<bool>& path, bool underscore=true);

    /**
     * ** helper functions and data for horizontal print **
     *  the width is 
     *   max(payload, width(left-subtree)) + 1 + width(right-subtree)
     *  the 1  in the middle is to separate the left from the right by one space
     *  if the pointer is NULL, then the width is 1
     *  here's how you can visualize the routine:
     *
     *  payload___
     *  |         \
     *  ---------- ----------
     *  | left   | | right  |
     *  | width  | | width  |
     *  ---------- ----------
     *
     *  or, in case the payload is really large
     *
     *  reallylargepayload
     *  |                 \
     *  ----------         ----------
     *  | left   |         | right  |
     *  | width  |         | width  |
     *  ----------         ----------
     *
     *  that's why we take max(payload, left-width) + 1 + right-width
     * ------------------------------------------------------------------------
     *  For printLevel(), we do the following
     *  given a vector of strings & their positions on a line, print them out
     *  at the correct positions; for example, if the input is
     *  [("abc", 0), ("xyz", 6), ("defg", 10)] then the routine prints
     *  abc   xyz defg
     *  ^     ^   ^
     *  0     6   10
     *  Note that
     *  vec[i].pos + vec[i].text.length() is the cursor position of the last
     *  character of the i'th string
     */
    void computeSubtreeWidths(Node*);
    std::string printLevel(const std::vector<Text>& vec);
    std::map<Node*, size_t> width_;

    /**
     * ** helper functions for symmetric print **
     * ------------------------------------------------------------------------
     *  computeCoordinates() computes both the width and the "midpoint" of the 
     *  box enclosing each subtree rooted at 'root'
     *  Base cases:
     *  - the width is 1 if the node is NULL, offset = 1
     *  - if both the left and the right children are NULL, then
     *    width = payload.length()
     *    offset = (payload.length()+1)/2
     *
     *  Inductive cases:
     *
     *  Case 1: payload + 2 + lo + r-ro + 1 <= l+r+1
     *
     *  ----------
     *            \
     *        __payload___
     *       /            \  <--- these are called "connectives"
     *  ----------    ----------
     *  | left   |    | right  |
     *  | width  |    | width  |
     *  ----------    ----------
     *  =====1     1  ======----     
     *   lo           ro    r-ro
     *
     *  the width is l+r+1
     *  new offset is lo + (l+r+1-(r-ro+1)-lo)/2 + 1
     *              = lo + (l+ro-lo)/2 + 1
     *
     *  Case 2: payload + lo + r-ro + 1 >= l+r
     *
     *  ----------
     *            \
     *        payloadlong
     *       /           \  <--- these are called "connectives"
     *  ----------   ----------
     *  | left   |   | right  |
     *  | width  |   | width  |
     *  ----------   ----------
     *  =====1    x  ======----     
     *   lo          ro    r-ro
     *
     *  here, we want l+r+x = payload + 2 + lo + r - ro + 1
     *  and hence x = payload+lo-ro+3-l
     *
     *  new offset is lo + (l+r+x-(r-ro+1)-lo)/2 + 1
     *              = lo + (l+ro-lo+x-1)/2 + 1
     *
     *  In both cases,
     *  x = max(payload+lo+r-ro+2,l+r) -l-r+1
     *  the width is l + x + r.
     *  and new offset is lo + (l+ro-lo+x-1)/2 + 1
     * ------------------------------------------------------------------------
     */
    void computeCoordinates(Node*);
    std::map<Node*, Coordinate> coordinate_;

    // Google Style guide indicates that a trailing underscore should be put
    // at the end of data member names
    Node* root_;
};

#endif
