function cp=clsu(pM,class,k1,k2,alp,beta,itnum)
%
%  clsu: combine the outputs of multiple supervised and unsupervised models
%        on a target set based on their consensus.
%             
%  Input: 
%       pM-a matrix containing all the base models' outputs on the target set. 
%          Each column represent one model. You must place classification models
%           before clustering models.
%
%       k1-number of classification models, so columns 1:k1 in the matrix pM are the
%          results of classification models
%
%       k2-number of clustering models, so columns k1+1:k1+k2 in the matrix pM
%          are the results of clustering models
%
%       class-the number of classes in the problem
%
%       itnum-number of iterations, usually set to 20
%
%       alp-parameter alpha in the paper, usually set to 0.4
%
%       beta-parameter beta in the paper, can tune from 0 to 1
%
%  Output:           
%       cp-the consensus label for all the examples in the target set
%
%==========================================================================
%
%
% An example:
%
%       %load the results of all the base models for Cora1 data set
%       directory='../data/cora/'; 
%       pM=load([directory,'mbase1.csv']); 
%
%       %get the consensus result
%       %the task has three classes, 2 classification models, 2 clustering
%       models. We set alp=0.4, beta=0.8 and itnum=20
%       cp=clsu(pM,3,2,2,0.4,0.8,20);
%

%step 1: propagation at the group level
[H,F]=propagate(pM,class,k1,k2,alp,itnum);

%step 2: get local weights for each example
W=weightall(pM,class);

%combine global and local consensus results to get the conditional probability
%of each example based on Eq. 2
pre=(beta*H+(1-beta)*H.*W)*F;

%for classification purpose, assign the class label with the highest probability 
%to each example  
[tmp,cp]=max(pre,[],2);

