/*
 * Copyright 1999 by Brown University Computer Science
 * 115 Waterman Street, 4th Floor, Providence, Rhode Island, 02906, U.S.A
 * All rights reserved.
 *
 * Permission is hereby granted for free use and modification of this
 * software.  However, this copyright must remain at all times.
 * Re-distribution is restricted to those with the express written consent
 * of the copyright holder.  Permission can be obtained by mailing
 * cs015headtas@cs.brown.edu.  
 */

package NGP;

/**
 * Provides a way to monitor keystrokes and respond accordingly.
 * Subclasses should override the {@link #keyPressed() keyPressed} method
 * to do something when the key is pressed.  It will need a reference
 * to an <code>NGP.Containers.Panel</code> so that it can "hear" when a 
 * key is pressed.<p>
 *
 * Sometimes you may need to press Tab to get the KeyInteractors 
 * working, we're not sure why (well, I think that's the way AWT determines
 * that a Panel has focus, but who knows).
 * 
 * @author Matt Chotin (<a href="mailto:mhc@cs.brown.edu">mhc</a>)
 */

public abstract class KeyInteractor implements java.awt.event.ActionListener {
	
  /** The actual object that hears the keypresses */
  private javax.swing.KeyStroke _stroke;
  /** The Panel that will tell the KeyStroke when something has occurred */
  private NGP.Containers.Panel _panel;
  /** 
   * Indicates whether the KeyInteractor should actually call keyPressed
   * when the key is pressed.
   */
  private boolean _enabled;
	
  /**
   * Create a <code>KeyInteractor</code> that will respond whenever the
   * key with the passed in character is pressed.
   * 
   * @param panel the Panel that will "hear" the keypresses, a good choice is
   * the highest level panel
   * @param keyChar the character on top of the key, pass ' ' for space, 'a'
   * for A without the Shift key and 'A' for A with the Shift key
   */
  public KeyInteractor(NGP.Containers.Panel panel, char keyChar) {
    _stroke = javax.swing.KeyStroke.getKeyStroke(keyChar);
    _panel = panel;
    this.enable();
  }
	
  /**
   * Turn the KeyInteractor on so that it will call 
   * {@link #keyPressed() keyPressed}.
   *
   * @see #disable() disable
   * @see #getEnabled() getEnabled
   */
  public void enable() {
    _enabled = true;
    _panel.registerKeyboardAction(this, _stroke, 
				  javax.swing.JComponent.WHEN_IN_FOCUSED_WINDOW);
  }
	
  /**
   * Turn the KeyInteractor off so that it <b>will not</b> call 
   * {@link #keyPressed() keyPressed}.
   *
   * @see #enable() enable
   * @see #getEnabled() getEnabled
   */
  public void disable() {
    _enabled = false;
    _panel.unregisterKeyboardAction(_stroke);
  }
	
  /**
   * Is this <code>KeyInteractor</code> enabled or disabled?
   *
   * @return <code>true</code> for enabled and <code>false</code> for disabled
   * @see #enable() enable
   * @see #disable() disable
   */
  public boolean getEnabled() {
    return _enabled;
  }
	
  /**
   * Subclasses should override to do something useful.  This will get called 
   * occasionally as long as the key is pressed.
   */
  public abstract void keyPressed();
	
  /**
   * Normal users need not deal with this method!
   * <p>
   * Called by the KeyStroke to indicate that it was pressed.  Calls our 
   * {@link #keyPressed() keyPressed} method.
   *
   * @param e we do nothing with this parameter
   */
  public void actionPerformed(java.awt.event.ActionEvent e) {
    this.keyPressed();
  }
		
}

